import * as assert from 'assert';
import * as express from 'express';
import {Request, Response, NextFunction} from 'express';
import * as Boom from 'boom';

import {logger} from './lib/logger';
import {intRouter} from './int/v1';

export const app = express()
    .disable('x-powered-by')
    .get('/ping', (_req, res) => res.end())
    .use('/int/v1', intRouter)
    .use((_req, _res, next) => next(Boom.notFound('Endpoint not found')))
    .use((err: any, req: Request, res: Response, next: NextFunction) => {
        if (Boom.isBoom(err)) {
            sendError(res, err);
        } else {
            logger.error(err.stack || err);
            sendError(res, Boom.internal());
        }
    });

if (!module.parent) {
    const port = getCustomPort() || 8080;
    app.listen(port, () => {
        logger.info(`Application started on port ${port}`);
    });
}

function sendError(res: Response, err: Boom): void {
    res.status(err.output.statusCode).json(err.output.payload);
}

function getCustomPort(): number | undefined {
    if (process.env.MAPS_NODEJS_PORT === undefined) {
        return;
    }
    const port = parseInt(process.env.MAPS_NODEJS_PORT, 10);
    assert(!isNaN(port), 'Environment variable MAPS_NODEJS_PORT must be an integer');
    return port;
}
