import {Agent as HttpAgent} from 'http';
import {Agent as HttpsAgent} from 'https';
import * as got from 'got';
import * as qs from 'qs';

import {config} from 'app/config';
import {logger} from './logger';

const httpAgent = new HttpAgent({keepAlive: true});
const httpsAgent = new HttpsAgent({keepAlive: true});

export interface RequestParams {
    origin: string;
    gotOptions: got.GotJSONOptions;
    qsOptions?: qs.IStringifyOptions;
    tvmTicket?: string;
}

export async function doRequest(url: string, {
    gotOptions,
    tvmTicket,
    origin: originalUrl,
    qsOptions
}: RequestParams): Promise<got.Response<any>> {
    const {query} = gotOptions;
    const method = gotOptions.method || 'GET';
    const params = qs.stringify(query, qsOptions) && `?${qs.stringify(query, qsOptions)}`;
    let logMessage = `${originalUrl} => ${method.toUpperCase()} => ${url}${params}`;

    if (gotOptions.headers) {
        logMessage += ` => (headers: ${JSON.stringify(gotOptions.headers)})`;
    }

    if (gotOptions.body) {
        logMessage += ` => (body: ${JSON.stringify(gotOptions.body)})`;
    }

    try {
        const response = await got(url, {
            ...config.request,
            ...gotOptions,
            headers: {
                ...gotOptions.headers,
                'x-ya-service-ticket': tvmTicket
            },
            agent: {
                http: httpAgent,
                https: httpsAgent
            }
        });

        logMessage += ` => ${response.statusCode}`;

        if (gotOptions.json) {
            logMessage += ` => ${JSON.stringify(response.body)}`;
        }

        logger.debug(logMessage);
        return response;
    } catch (error) {
        logger.error(
            `${logMessage} => ${error.statusCode || error.code}${getErrorBody(error)}`
        );
        throw error;
    }
}

function getErrorBody(error: any): string {
    if (error.response && error.response.body) {
        return ` (message: ${JSON.stringify(error.response.body)}))`;
    }

    return '';
}
