import { IncomingMessage, ServerResponse } from 'node:http';

import { RequestHandlerOptions } from 'types';

import { HttpStatusCode } from 'consts/HttpStatusCode';

import { fetchS3 } from 'lib/api/fetchS3/fetchS3';
import { getLandingCSP } from 'lib/csp/getLandingCSP';
import { sendResHtml } from 'lib/response/sendResHtml/sendResHtml';
import { setResNoCacheHeaders } from 'lib/response/setResNoCacheHeaders/setResNoCacheHeaders';
import { setResSecurityHeaders } from 'lib/response/setResSecurityHeaders/setResSecurityHeaders';

export async function landingPageController(
    req: IncomingMessage,
    res: ServerResponse,
    { logger }: RequestHandlerOptions,
): Promise<void> {
    const content = await fetchS3('landing/index.html', { ignorePrefix: true, req, logger });

    if (content.statusCode === HttpStatusCode.NOT_FOUND) {
        return sendResHtml(res, HttpStatusCode.NOT_FOUND, 'Frontend not found');
    } else if (content.statusCode !== HttpStatusCode.OK) {
        return sendResHtml(res, HttpStatusCode.INTERNAL_SERVER_ERROR, 'HTTP500: Frontend Server Error');
    }

    const csp = getLandingCSP(req);

    setResSecurityHeaders(res, csp, false);
    setResNoCacheHeaders(res);

    sendResHtml(res, HttpStatusCode.OK, content.body);
}
