import { Logger } from '@yandex-int/yandex-logger';
import got from 'got';

import { HttpStatusCode } from 'consts/HttpStatusCode';

import { code2level } from 'lib/logger/code2level';

export interface CreateStartrekTicketData {
    type: 'task';
    queue: string;
    summary: string;
    description: string;
}

export interface CreateStartrekTicketOptions {
    logger: Logger;
}

const REQUEST_TIMEOUT = 5000;

export async function createStartrekTicket(
    ticket: CreateStartrekTicketData,
    { logger }: CreateStartrekTicketOptions,
): Promise<string> {
    logger.info('[StartrekReq] POST /v2/issues');

    const response = await got.post('https://st-api.yandex-team.ru/v2/issues', {
        json: ticket,
        headers: {
            Authorization: 'OAuth ' + process.env.DM_STARTREK_TOKEN,
        },
        timeout: {
            request: REQUEST_TIMEOUT,
        },
        throwHttpErrors: false,
    });

    let ticketKey: string | undefined = undefined;
    let errors: string[] | undefined = undefined;

    let resposeData: any | undefined = undefined;

    try {
        resposeData = JSON.parse(response.body);
    } catch (error) {
        if (error instanceof Error) {
            errors = [error.message];
        }
    }

    if (resposeData) {
        if (response.statusCode === HttpStatusCode.CREATED) {
            ticketKey = resposeData.key;
        } else {
            errors = resposeData.errorMessages;
        }
    }

    logger[code2level(response.statusCode)](
        { info: { statusCode: response.statusCode, ticketKey, errors, timings: response.timings } },
        '[StartrekRes] POST /v2/issues',
    );

    if (!ticketKey) {
        throw new Error('Startrek ticket not created');
    }

    return ticketKey;
}
