import { IncomingMessage } from 'node:http';

import { Logger } from '@yandex-int/yandex-logger';
import got from 'got';

import { code2level } from 'lib/logger/code2level';
import { getReqHost } from 'lib/request/getReqHost/getReqHost';
import { getReqHostService, HostService } from 'lib/request/getReqHostService/getReqHostService';
import { getReqId } from 'lib/request/getReqId/getReqId';

const DM_PROFILE = process.env.DM_PROFILE;

const API_ENDPOINT: Record<HostService, string> = {
    drivematics:
        DM_PROFILE === 'test' ? 'https://testing.drivematics.yandex.ru/api/' : 'https://drivematics.yandex.ru/api/',
    vlootkit: 'https://vlootkit.com/api/',
};

// eslint-disable-next-line @typescript-eslint/no-magic-numbers
const REQUEST_TIMEOUT = process.env.DM_API_TIMEOUT ? Number(process.env.DM_API_TIMEOUT) : 1500;

export interface FetchApiOptions {
    req: IncomingMessage;
    logger: Logger;
}

export async function fetchApi(path: string, { req, logger }: FetchApiOptions) {
    const service = getReqHostService(req);
    const host = getReqHost(req);
    const reqId = getReqId(req) || 'unknown';

    logger.info({ info: { host, path } }, '[ApiReq] %s', path);

    const proxyHeaders = Object.entries(req.headers).reduce((memo: Record<string, string | string[]>, [key, value]) => {
        if (key.startsWith('x-') && value) {
            memo[key] = value;
        }

        return memo;
    }, {});

    const response = await got(API_ENDPOINT[service] + path + '?reqid=' + encodeURIComponent(reqId), {
        headers: {
            ...proxyHeaders,
            'x-forwarded-host': host,
            cookie: req.headers.cookie,
        },
        timeout: {
            request: REQUEST_TIMEOUT,
        },
        throwHttpErrors: false,
    });

    logger[code2level(response.statusCode)](
        { info: { host, path, statusCode: response.statusCode, timings: response.timings } },
        '[ApiRes] %s',
        path,
    );

    return response;
}
