import { IncomingMessage } from 'node:http';

import { Logger } from '@yandex-int/yandex-logger';

import { HttpStatusCode } from 'consts/HttpStatusCode';

import { fetchS3 } from 'lib/api/fetchS3/fetchS3';
import { getReqHostService } from 'lib/request/getReqHostService/getReqHostService';
import { getReqLang } from 'lib/request/getReqLang/getReqLang';

export interface FetchPageOptions {
    req: IncomingMessage;
    logger: Logger;
    ignorePrefix?: boolean;
}

const DM_PROFILE = process.env.DM_PROFILE;

export async function fetchPage(path: string, { ignorePrefix, req, logger }: FetchPageOptions) {
    const service = getReqHostService(req);
    const lang = getReqLang(req);

    let content;

    if (DM_PROFILE === 'prod') {
        // cdn version
        content = await fetchS3(`pages/${service}-cdn/${lang}/${path}`, { ignorePrefix, req, logger });

        if (content.statusCode !== HttpStatusCode.NOT_FOUND) {
            return content;
        }
    }

    // none cdn version
    content = await fetchS3(`pages/${service}/${lang}/${path}`, { ignorePrefix, req, logger });

    if (content.statusCode !== HttpStatusCode.NOT_FOUND) {
        return content;
    }

    // fallback to unified service page
    content = await fetchS3(`pages/${lang}/${path}`, { ignorePrefix, req, logger });

    if (content.statusCode !== HttpStatusCode.NOT_FOUND) {
        return content;
    }

    // fallback to eng
    content = await fetchS3(`pages/en/${path}`, { ignorePrefix, req, logger });

    if (content.statusCode !== HttpStatusCode.NOT_FOUND) {
        return content;
    }

    // fallback to legacy
    return fetchS3(path, { ignorePrefix, req, logger });
}
