import { IncomingMessage } from 'node:http';

import { Logger } from '@yandex-int/yandex-logger';
import got from 'got';

import { code2level } from 'lib/logger/code2level';
import { getReqHost } from 'lib/request/getReqHost/getReqHost';
import { getReqHostService, HostService } from 'lib/request/getReqHostService/getReqHostService';

const DM_PROFILE = process.env.DM_PROFILE;

const REQUEST_TIMEOUT = 1000;

const API_ENDPOINT: Record<HostService, string> = {
    drivematics:
        DM_PROFILE === 'test' ? 'https://drivematics-test.s3.mds.yandex.net/' : 'https://drivematics.s3.yandex.net/',
    vlootkit: 'https://vlootkit-frontend.s3.eu-central-1.amazonaws.com/',
};

const subhostRegExp = /^(crowd|pr-[0-9]+)\./;

export interface FetchS3Options {
    req: IncomingMessage;
    logger: Logger;
    ignorePrefix?: boolean;
}

function bucketPrefix(req: IncomingMessage): string {
    if (DM_PROFILE === 'test') {
        const match = getReqHost(req).match(subhostRegExp);
        if (match && match[1]) {
            // pr base route
            return match[1] + '/web/';
        }

        // test route
        return 'test/web/';
    }

    // prod
    return '';
}

export async function fetchS3(path: string, { ignorePrefix, req, logger }: FetchS3Options) {
    const service = getReqHostService(req);
    const host = getReqHost(req);
    const fullPath = (ignorePrefix ? '' : bucketPrefix(req)) + path;

    logger.info({ info: { host, path: fullPath } }, '[S3Req] %s', fullPath);

    const response = await got(API_ENDPOINT[service] + fullPath, {
        timeout: {
            request: REQUEST_TIMEOUT,
        },
        throwHttpErrors: false,
    });

    logger[code2level(response.statusCode)](
        { info: { host, path: fullPath, statusCode: response.statusCode, timings: response.timings } },
        '[S3Res] %s',
        fullPath,
    );

    return response;
}
