import { IncomingMessage } from 'node:http';

import { parse } from 'cookie';

import { getReqHeaderStr } from 'lib/request/getReqHeaderStr/getReqHeaderStr';
import { getReqHostTld } from 'lib/request/getReqHostTld/getReqHostTld';

export type ReqLang = 'en' | 'cs' | 'ru';

const LANGS = {
    en: true,
    cs: true,
};

const DEFAULT_LANG = 'en';
const MAX_HEADER_SIZE = 1024;

export function getReqLang(req: IncomingMessage): ReqLang {
    const cookies = parse(getReqHeaderStr(req, 'cookie') || '');

    const tld = getReqHostTld(req);
    const allowedLangs = tld === 'ru' ? { ...LANGS, ru: true } : LANGS;

    if (cookies && cookies.dm_lang && allowedLangs.hasOwnProperty(cookies.dm_lang)) {
        return cookies.dm_lang as ReqLang;
    }

    if (cookies && cookies.vlang && allowedLangs.hasOwnProperty(cookies.vlang)) {
        return cookies.vlang as ReqLang;
    }

    const langs = getReqHeaderStr(req, 'accept-language');

    const parts: string[] = langs && langs.length < MAX_HEADER_SIZE ? langs.split(',') : [];

    const items: Array<{ lang: string; code: string; q: number }> = parts
        .map((item) => {
            const parts = item.toLowerCase().split(';q=');
            const code = parts[0].trim();
            const q = parts[1] ? Number(parts[1]) || 1 : 1;
            const langParts = code.split('-');

            return {
                lang: langParts[0].trim(),
                code,
                q,
            };
        })
        .sort((a, b) => {
            return b.q - a.q;
        });

    const item = items.find((item) => {
        return allowedLangs.hasOwnProperty(item.lang);
    });

    return item ? (item.lang as ReqLang) : DEFAULT_LANG;
}
