import { createServer, IncomingMessage, ServerResponse } from 'node:http';
import { performance } from 'node:perf_hooks';

import { errorPageController } from 'controllers/errorPageController';
import { forbiddenPageController } from 'controllers/forbiddenPageController';
import { formController } from 'controllers/formController';
import { landingPageController } from 'controllers/landingPageController';
import { versionController } from 'controllers/versionController';
import { webPageController } from 'controllers/webPageController';
import { parse } from 'cookie';
import { RequestHandlerOptions } from 'types';

import { HttpStatusCode } from 'consts/HttpStatusCode';

import { fetchApi } from 'lib/api/fetchApi/fetchApi';
import { code2level } from 'lib/logger/code2level';
import { logger } from 'lib/logger/logger';
import { getReqHeaderStr } from 'lib/request/getReqHeaderStr/getReqHeaderStr';
import { getReqHost } from 'lib/request/getReqHost/getReqHost';
import { getReqHostService } from 'lib/request/getReqHostService/getReqHostService';
import { getReqHostTld } from 'lib/request/getReqHostTld/getReqHostTld';
import { getReqId } from 'lib/request/getReqId/getReqId';
import { sendResHtml } from 'lib/response/sendResHtml/sendResHtml';
import { sendResRedirect } from 'lib/response/sendResRedirect/sendResRedirect';

const DEFAULT_PORT = 9000;

const DM_PORT = process.env.DM_PORT || DEFAULT_PORT;

async function requestHandler({
    req,
    res,
    options,
}: {
    req: IncomingMessage;
    res: ServerResponse;
    options: RequestHandlerOptions;
}): Promise<void> {
    const { logger } = options;

    // HACK: feedback form
    if (req.method === 'POST' && req.url === '/web-api/request/demo') {
        return formController(req, res, options);
    }

    // non get requests guard
    if (req.method !== 'GET' && req.method !== 'HEAD') {
        return sendResHtml(res, HttpStatusCode.METHOD_NOT_ALLOWED, 'HTTP405: Method Not Allowed');
    }

    // favicon fix
    if (req.url === '/favicon.ico') {
        return sendResHtml(res, HttpStatusCode.NOT_FOUND, 'HTTP404: Not Found');
    }

    // health check
    if (req.url === '/version.json') {
        return versionController(req, res, options);
    }

    // normal logic here
    const response = await fetchApi('leasing/login2', { req, logger });
    const backendStatusCode = response.statusCode;

    if (backendStatusCode === HttpStatusCode.OK) {
        // auth
        return webPageController(req, res, options);
    } else if (backendStatusCode === HttpStatusCode.UNAUTHORIZED || backendStatusCode === HttpStatusCode.FORBIDDEN) {
        const host = getReqHost(req);
        const service = getReqHostService(req);
        const url = new URL(req.url || '', 'https://' + host);

        // unauth
        if (url.pathname === '/') {
            if (service === 'vlootkit') {
                if (backendStatusCode === HttpStatusCode.FORBIDDEN) {
                    return forbiddenPageController(req, res, options);
                }

                const cookies = parse(getReqHeaderStr(req, 'cookie') || '');
                if (cookies.vreftoken) {
                    return sendResRedirect(res, `/auth?retpath=${encodeURIComponent('/')}`);
                }
            }

            // landing page only for root
            return landingPageController(req, res, options);
        } else {
            if (backendStatusCode === HttpStatusCode.FORBIDDEN) {
                return forbiddenPageController(req, res, options);
            }

            if (service === 'drivematics') {
                const tld = getReqHostTld(req);
                const retpath = encodeURIComponent('https://' + host + req.url);
                return sendResRedirect(res, `https://passport.yandex.${tld}/auth/list?retpath=${retpath}&noreturn=1`);
            } else {
                const retpath = encodeURIComponent(req.url || '/');
                return sendResRedirect(res, `/auth?retpath=${retpath}`);
            }
        }
    } else {
        // unexpected http code
        return errorPageController(req, res, options);
    }
}

const server = createServer((req, res) => {
    const startTime = performance.now();
    const reqId = getReqId(req) || '-';

    const requestLogger = logger.child({
        reqId,
        host: getReqHost(req),
    });

    requestLogger.info('[Req] %s %s', req.method || 'UNKNOWN', req.url);

    requestHandler({ req: req, res: res, options: { logger: requestLogger } })
        .catch((error) => {
            requestLogger.error(error, 'Request handler error');

            return errorPageController(req, res, { logger: requestLogger });
        })
        .catch((error) => {
            requestLogger.error(error, 'Impossible request handler error');
        })
        .finally(() => {
            requestLogger[code2level(res.statusCode)](
                '[Res] %s %s %d %sms',
                req.method || 'UNKNOWN',
                req.url,
                res.statusCode,
                performance.now() - startTime,
            );
        });
});

server.listen(DM_PORT, () => {
    logger.info(`Server running at http://localhost:${DM_PORT}/`);
});
