const { awacs, Provider: AwacsProvider } = require('@yandex-int/awacs-api');
const tokenator = require('@yandex-int/tokenator');

const { logger } = require('../lib/logger');

async function getAwacsToken() {
    let awacsToken = process.env.AWACS_TOKEN;

    if (!awacsToken) {
        let tokenatorTokens = await tokenator(['awacs']);
        awacsToken = tokenatorTokens.awacs;
    }

    return awacsToken;
}

async function updateUpstream(awacs, name, spec) {
    let upstreamMeta = getUpstreamMeta(name);
    let upstreamData;

    try {
        let result = await awacs.getUpstream(upstreamMeta);
        upstreamData = result.upstream;
    } catch (e) {
        // noop
    }

    if (!upstreamData) {
        await awacs.createUpstream({ meta: upstreamMeta, spec: spec });
    } else {
        await awacs.updateUpstream({ meta: upstreamData.meta, spec: spec });
    }
}

function getUpstreamMeta(name) {
    return {
        namespaceId: 'testing.fleetmatic.yandex.ru',
        id: name,
        auth: {
            type: awacs.model.Auth.AuthType.STAFF,
            staff: {
                owners: {
                    logins: ['next0', 'robot-krtek'],
                    groupIds: [],
                },
            },
        },
    };
}

function getStaticUpstreamSpec(slug) {
    let yaml = `
l7_upstream_macro:
  version: 0.0.1
  id: ${slug}
  matcher:
    and_:
    - host_re: ${slug}\\.(testing|prestable|prod)\\.drivematics\\.yandex\\.(ru|com)
    - path_re: /static(/.*)?
  rewrite:
  - target: PATH
    pattern:
      re: '/static/(.*)?'
    replacement: '/${slug}/web/static/%1'
  headers:
  - rewrite:
      target: 'Host'
      pattern:
        re: '(.*)'
      replacement: 'drivematics-test.s3.mds.yandex.net'
  flat_scheme:
    balancer:
      attempts: 2
      max_reattempts_share: 0.15
      max_pessimized_endpoints_share: 0.20
      fast_attempts: 2
      retry_http_responses:
        codes:
        - 5xx
        on_last_failed_retry: PROXY_RESPONSE_AS_IS
      connect_timeout: 70ms
      backend_timeout: 5s
    backend_ids:
    - drivematics_test_s3
    on_error:
      static:
        status: 504
        content: Service unavailable
    `.trim();

    return {
        labels: {
            order: '01000000',
        },
        type: awacs.model.BalancerType.YANDEX_BALANCER,
        yandexBalancer: {
            mode: awacs.model.YandexBalancerUpstreamSpec.ConfigMode.EASY_MODE2,
            yaml,
        },
    };
}

async function main() {
    let prNumber = Number(process.env.CI_PULL_REQUEST_ID);

    if (!prNumber) {
        throw new Error('CI_PULL_REQUEST_ID env variable is not specified');
    }

    let awacsToken = await getAwacsToken();
    let awacs = new AwacsProvider({ token: awacsToken });

    let slug = 'pr-' + prNumber;

    try {
        await updateUpstream(awacs, slug, getStaticUpstreamSpec(slug));

        logger.info(`AWACS ${slug} configured successfully`);
        logger.info('Please wait ~10min to deploy configuration for all balancers');
        logger.info(`Testing stand link: https://${slug}.testing.drivematics.yandex.ru/`);
        logger.info(`Prestable stand link: https://${slug}.prestable.drivematics.yandex.ru/`);
        logger.info(`Prod stand link: https://${slug}.prod.drivematics.yandex.ru/`);
    } catch (e) {
        logger.error(`AWACS ${slug} beta failed`);

        throw e;
    }
}

main().catch((error) => {
    logger.error(error);
    process.exit(1);
});
