#!/usr/bin/env node

const https = require('https');
const fs = require('fs');
const path = require('path');
const crypto = require('crypto');

const SANDBOX_OAUTH_TOKEN = process.env.SANDBOX_OAUTH_TOKEN;

const content = ['../../package-lock.json']
    .map((file) => path.resolve(__dirname, file))
    .map((file) => fs.readFileSync(file, 'utf8'))
    .join('');

const packageJsonMd5 = crypto.createHash('md5').update(content).digest('hex');

function fetch(url, options) {
    return new Promise((resolve, reject) => {
        https
            .get(url, options, (res) => {
                if (res.statusCode !== 200) {
                    return reject(new Error('Status code error: HTTP' + res.statusCode));
                }

                let data = [];

                res.on('data', (chunk) => {
                    data.push(chunk);
                });

                res.on('end', () => {
                    try {
                        const payload = JSON.parse(Buffer.concat(data).toString());

                        resolve(payload);
                    } catch (error) {
                        reject(error);
                    }
                });
            })
            .on('error', (error) => {
                reject(error);
            });
    });
}

(async () => {
    try {
        const headers = {
            Authorization: 'OAuth ' + SANDBOX_OAUTH_TOKEN,
            Accept: 'application/json; charset=utf-8',
        };

        const query = {
            type: 'NODEJS_MODULE',
            owner: 'DRIVEMATICS_FRONTEND',
            state: 'READY',
            attrs: JSON.stringify({
                app: 'drivematics',
                package_json_md5: packageJsonMd5,
            }),
            limit: 1,
            offset: 0,
        };

        const queryStr = Object.entries(query)
            .map(([key, value]) => {
                return key + '=' + encodeURIComponent(value);
            })
            .join('&');

        const result = await fetch('https://sandbox.yandex-team.ru/api/v1.0/resource?' + queryStr, {
            headers,
            timeout: 5000,
        });

        let resourceId;

        if (result && Array.isArray(result.items) && result.items.length) {
            resourceId = result.items[0].id;
        }

        process.stdout.write(
            JSON.stringify({
                id: resourceId,
                package_json_md5: packageJsonMd5,
            }),
        );
    } catch (error) {
        console.error(error);
        process.exit(1);
    }
})();
