const path = require('path');

const ERROR_MESSAGE_ID = 'errorMessage';
const messages = {
    [ERROR_MESSAGE_ID]: "Relative imports are not allowed. Use '{{expectedPath}}' instead of '{{source}}'.",
};

const create = (context) => {
    function checkNode(node) {
        const { baseUrl, excludes } = context.options[0];

        const source = node.source.value;

        if (source.startsWith('.') && !source.includes(excludes)) {
            const filename = context.getFilename();
            const absolutePath = path.normalize(path.join(path.dirname(filename), source));
            const expectedPath = path.relative(baseUrl, absolutePath);

            if (source !== expectedPath) {
                context.report({
                    node,
                    messageId: ERROR_MESSAGE_ID,
                    data: { expectedPath, source },
                    fix: function (fixer) {
                        return fixer.replaceText(node.source, `'${expectedPath}'`);
                    },
                });
            }
        }
    }

    return {
        ImportDeclaration: checkNode,
        ImportExpression: checkNode,
        ExportAllDeclaration: checkNode,
    };
};

module.exports = {
    create,
    name: 'only-absolute-paths',
    meta: {
        type: 'suggestion',
        docs: {
            description: 'Absolute paths',
        },
        fixable: true,
        schema: [
            {
                type: 'object',
                properties: {
                    baseUrl: {
                        type: 'string',
                    },
                    excludes: {
                        type: 'array',
                        items: {
                            type: 'string',
                        },
                    },
                },
                additionalProperties: false,
            },
        ],
        messages,
    },
};
