import tokenator from '@yandex-int/tokenator';

import { MovaTranslateValue, MovaWatcherTranslatesStore } from 'mova-watcher/dist/types';

import { post } from './fetch';
import { TankerKeysetTranslate, TankerKeysetTranslatePayload, TankerLang } from './types';
import { LANGS, LANGS_PLURALS } from './consts';

interface TankerKeysetItem {
    name: string;
    translations: Record<TankerLang, TankerKeysetTranslate>;
    meta: {
        context: string;
        plural: boolean;
    };
}

interface TankerStore {
    keys: TankerKeysetItem[];
}

function mova2tanker(t: MovaTranslateValue, lang: TankerLang): TankerKeysetTranslatePayload {
    const translate = t[LANGS[lang]];

    // singular
    if (typeof translate === 'string' || typeof translate === 'undefined') {
        return { singular_form: translate || '' };
    }

    // plural
    return Object.keys(LANGS_PLURALS[lang]).reduce((memo, key) => {
        memo[key] = (translate && translate[LANGS_PLURALS[lang][key]]) || '';

        return memo;
    }, {});
}

export async function exporter(store: MovaWatcherTranslatesStore): Promise<void> {
    const tokens = await tokenator(['tanker']);
    const tankerToken = tokens.tanker;

    const keys: TankerKeysetItem[] = [];

    for (const key of Object.keys(store)) {
        let groupsCount = 0;

        for (let i = 0; i < store[key].length; i++) {
            const value = store[key][i];
            const plural = Boolean(value.t.en && typeof value.t.en !== 'string');

            if (value.t._meta?.group) {
                groupsCount += 1;
            }

            keys.push({
                name: [key, value.t._meta?.group ? ` [#${value.t._meta?.group}]` : ''].join(''),
                translations: Object.keys(LANGS).reduce((memo, lang: TankerLang) => {
                    memo[lang] = {
                        language: lang,
                        status: value.t._meta?.status?.[LANGS[lang]] || 'REQUIRES_TRANSLATION',
                        payload: mova2tanker(value.t, lang),
                    };

                    return memo;
                }, {}),
                meta: {
                    context: [value.t._meta?.context, '-----', 'hash: ' + value.hash, ...value.files]
                        .filter(Boolean)
                        .join('\n'),
                    plural,
                },
            } as TankerKeysetItem);
        }

        if (store[key].length - groupsCount > 1) {
            console.log('[ERROR] no `_meta.group`', store[key]);

            throw new Error('[mova-i18n] exporter fatal error');
        }
    }

    const tanker: TankerStore = { keys };

    const res = await post(
        'https://tanker.yandex-team.ru/api/v1/project/Drivematics/keyset/frontend_web/import?mode=CREATE_OR_UPDATE_KEYS_DROP_OTHERS',
        tanker,
        {
            headers: {
                Authorization: 'OAuth ' + tankerToken,
            },
        },
    );

    console.log(res);
}
