import type { IncomingMessage } from 'http';
import https, { RequestOptions } from 'https';

function buildResponseHandler(resolve: (data: any) => void, reject: (error: Error) => void) {
    return (res: IncomingMessage) => {
        let data: any[] = [];

        res.on('data', (chunk) => {
            data.push(chunk);
        });

        res.on('end', () => {
            const payload: string = Buffer.concat(data).toString();

            if (res.statusCode !== 200) {
                return reject(new Error('Status code error: HTTP' + res.statusCode + '\n' + payload));
            }

            try {
                resolve(JSON.parse(payload));
            } catch (error) {
                reject(error);
            }
        });
    };
}

export function fetch<T = unknown>(url, options) {
    return new Promise<T>((resolve, reject) => {
        https.get(url, options, buildResponseHandler(resolve, reject)).on('error', reject);
    });
}

export function post<T = unknown>(url: string, data: any, options: RequestOptions) {
    const payload = JSON.stringify(data);

    const httpOptions: RequestOptions = {
        ...options,
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Content-Length': Buffer.byteLength(payload, 'utf-8'),
            ...options.headers,
        },
    };

    return new Promise<T>((resolve, reject) => {
        const req = https.request(url, httpOptions, buildResponseHandler(resolve, reject)).on('error', reject);

        req.end(payload, 'utf-8');
    });
}
