import tokenator from '@yandex-int/tokenator';

import { MovaLang, MovaTranslate, MovaTranslateValue, MovaWatcherTranslatesStore } from 'mova-watcher/dist/types';

import { fetch } from './fetch';
import { TankerKeysetTranslate, TankerLang } from './types';
import { LANGS_PLURALS, MOVA_LANGS } from './consts';

interface TankerKeysetItem {
    name: string;
    translations: Record<TankerLang, TankerKeysetTranslate>;
    meta: {
        context: {
            text: string;
        };
        plural: boolean;
    };
}

interface TankerStore {
    keysets: {
        frontend_web: {
            keys: Record<string, TankerKeysetItem>;
        };
    };
}

function trim(value: string | undefined): string | undefined {
    return typeof value === 'string' ? value.trim() : value;
}

function tanker2mova(t: TankerKeysetItem, lang: TankerLang): MovaTranslate {
    const payload = t.translations[lang].payload;

    if (t.meta.plural) {
        return Object.keys(LANGS_PLURALS[lang]).reduce((memo, form) => {
            memo[LANGS_PLURALS[lang][form]] = trim(payload[form]);

            return memo;
        }, {});
    } else {
        return trim(payload.singular_form || '');
    }
}

export async function importer(langs: MovaLang[]): Promise<MovaWatcherTranslatesStore> {
    const tokens = await tokenator(['tanker']);
    const tankerToken = tokens.tanker;

    const res = await fetch<TankerStore>(
        'https://tanker.yandex-team.ru/api/v1/project/Drivematics/keyset/frontend_web/export?sort=name&use_fall_backs=false',
        {
            headers: {
                Authorization: 'OAuth ' + tankerToken,
            },
        },
    );

    const data = res.keysets.frontend_web.keys;
    const keys = Object.keys(data);

    const store: MovaWatcherTranslatesStore = {};

    for (const key of keys) {
        const normalizedKey = key.replace(/\s+\[#[^\]]*\]$/, '');

        if (!store.hasOwnProperty(normalizedKey)) {
            store[normalizedKey] = [];
        }

        const t: MovaTranslateValue = { _meta: {} };
        for (const movaLang of langs) {
            const lang = MOVA_LANGS[movaLang];

            if (!t._meta!.status) {
                t._meta!.status = {};
            }

            if (data[key].translations[lang]) {
                t._meta!.status[movaLang] = data[key].translations[lang].status;
                t[movaLang] = tanker2mova(data[key], lang);
            } else {
                console.log('[ERROR] no lang found', lang, data[key]);

                throw new Error('[mova-i18n] importer fatal error');
            }
        }

        let hash: string | undefined;
        const contextParts = data[key].meta.context.text.match(/hash:\s+([a-z0-9]+)/);

        if (contextParts && contextParts[1]) {
            hash = contextParts[1];
        }

        if (hash) {
            store[normalizedKey].push({
                hash,
                t,
                files: [],
            });
        } else {
            console.log('[ERROR] no hash value', data[key]);

            throw new Error('[mova-i18n] importer fatal error');
        }
    }

    return store;
}
