'use strict';

var t = require('@babel/types');
var node_crypto = require('node:crypto');

function _interopNamespace(e) {
    if (e && e.__esModule) return e;
    var n = Object.create(null);
    if (e) {
        Object.keys(e).forEach(function (k) {
            if (k !== 'default') {
                var d = Object.getOwnPropertyDescriptor(e, k);
                Object.defineProperty(
                    n,
                    k,
                    d.get
                        ? d
                        : {
                              enumerable: true,
                              get: function () {
                                  return e[k];
                              },
                          },
                );
            }
        });
    }
    n['default'] = e;
    return Object.freeze(n);
}

var t__namespace = /*#__PURE__*/ _interopNamespace(t);

const HASH_PREFIX = '$';
function hash(str, len = 4) {
    return HASH_PREFIX + node_crypto.createHash('sha1').update(str).digest('hex').slice(0, len);
}

function getObjectPropertyValue(prop) {
    if (
        t__namespace.isObjectProperty(prop) &&
        (t__namespace.isIdentifier(prop.key) || t__namespace.isStringLiteral(prop.key))
    ) {
        return t__namespace.isIdentifier(prop.key) ? prop.key.name : prop.key.value;
    }
    return undefined;
}
function hasTranslatedValue(prop) {
    if (!prop) {
        return false;
    }
    const node = prop.value;
    if (t__namespace.isStringLiteral(node)) {
        return Boolean(node.value);
    } else if (t__namespace.isObjectExpression(node)) {
        return node.properties.some((prop) => {
            return (
                t__namespace.isObjectProperty(prop) &&
                t__namespace.isStringLiteral(prop.value) &&
                Boolean(prop.value.value)
            );
        });
    }
    return false;
}
function processTranslates(node, options) {
    const { lang, langPriorityList, hashLen } = options;
    return node.properties.forEach((prop) => {
        const keyName = getObjectPropertyValue(prop);
        if (
            t__namespace.isObjectProperty(prop) &&
            t__namespace.isObjectExpression(prop.value) &&
            keyName &&
            // don't apply mutation to hashed value
            !keyName.startsWith(HASH_PREFIX)
        ) {
            prop.key = t__namespace.identifier(hash(keyName, hashLen));
            // leave only one target language
            const cache = new Map();
            for (const langProp of prop.value.properties) {
                const langPropName = getObjectPropertyValue(langProp);
                if (t__namespace.isObjectProperty(langProp) && langPropName) {
                    cache.set(langPropName, langProp);
                }
            }
            let langProp;
            for (const lang of langPriorityList) {
                const currentLangProp = cache.get(lang);
                if (hasTranslatedValue(currentLangProp)) {
                    langProp = currentLangProp;
                    break;
                }
            }
            prop.value.properties = langProp
                ? [t__namespace.objectProperty(t__namespace.identifier(lang), langProp.value)]
                : [];
        }
    });
}

function plugin() {
    const DEFAULT_HASH_LEN = 4;
    const DEFAULT_LANG = process.env.LANG || 'en';
    return {
        name: 'mova-babel-plugin-i18n',
        visitor: {
            // process.env.LANG -> opts.lang
            MemberExpression(path, state) {
                if (path.get('object').matchesPattern('process.env')) {
                    const node = path.node;
                    if (t__namespace.isIdentifier(node.property) && node.property.name === 'LANG') {
                        path.replaceWith(t__namespace.stringLiteral(state.opts.lang || DEFAULT_LANG));
                    }
                }
            },
            // store file
            VariableDeclarator(path, state) {
                const lang = state.opts.lang || DEFAULT_LANG;
                const fallbackLangs = state.opts.fallbackLangs || {};
                const hashLen = state.opts.hashLen || DEFAULT_HASH_LEN;
                const langPriorityList = [lang, ...(fallbackLangs[lang] || [])];
                if (state.filename.endsWith('.i18n.ts')) {
                    const node = path.node;
                    if (t__namespace.isIdentifier(node.id) && node.id.name === 'translates') {
                        const init = node.init;
                        const processOptions = {
                            lang,
                            langPriorityList,
                            hashLen,
                        };
                        if (
                            t__namespace.isCallExpression(init) &&
                            t__namespace.isIdentifier(init.callee) &&
                            init.callee.name === 'keyset' &&
                            init.arguments.length &&
                            t__namespace.isObjectExpression(init.arguments[0])
                        ) {
                            // parse case: `const translates = keyset({ ... });`
                            processTranslates(init.arguments[0], processOptions);
                        } else if (t__namespace.isObjectExpression(init)) {
                            // parse case: `const translates = { ... };`
                            processTranslates(init, processOptions);
                        }
                    }
                }
            },
            // source file
            ImportDeclaration(path, state) {
                if (!state.filename.endsWith('.i18n.ts')) {
                    const node = path.node;
                    const importPath = node.source.value;
                    if (importPath.endsWith('.i18n')) {
                        const specs = node.specifiers;
                        const spec =
                            specs.length === 1 && t__namespace.isImportSpecifier(specs[0]) ? specs[0] : undefined;
                        if (
                            spec &&
                            spec.imported &&
                            t__namespace.isIdentifier(spec.imported) &&
                            spec.imported.name === 'i18n'
                        ) {
                            // import { i18n } from 'path/to/Component/Component.i18n';
                            this._movaTransformationEnabled = true;
                        }
                    }
                }
            },
            CallExpression(path, state) {
                const hashLen = state.opts.hashLen || 4;
                if (!state.filename.endsWith('.i18n.ts') && this._movaTransformationEnabled) {
                    const node = path.node;
                    if (t__namespace.isIdentifier(node.callee) && node.callee.name === 'i18n') {
                        if (node.arguments.length && t__namespace.isStringLiteral(node.arguments[0])) {
                            const value = node.arguments[0].value;
                            // don't apply mutation to hashed value
                            if (!value.startsWith(HASH_PREFIX)) {
                                node.arguments[0].value = hash(value, hashLen);
                            }
                        }
                    }
                }
            },
        },
    };
}

module.exports = plugin;
