import * as t from '@babel/types';
import { createHash } from 'node:crypto';

const HASH_PREFIX = '$';
function hash(str, len = 4) {
    return HASH_PREFIX + createHash('sha1').update(str).digest('hex').slice(0, len);
}

function getObjectPropertyValue(prop) {
    if (t.isObjectProperty(prop) && (t.isIdentifier(prop.key) || t.isStringLiteral(prop.key))) {
        return t.isIdentifier(prop.key) ? prop.key.name : prop.key.value;
    }
    return undefined;
}
function hasTranslatedValue(prop) {
    if (!prop) {
        return false;
    }
    const node = prop.value;
    if (t.isStringLiteral(node)) {
        return Boolean(node.value);
    }
    else if (t.isObjectExpression(node)) {
        return node.properties.some((prop) => Boolean(getObjectPropertyValue(prop)));
    }
    return false;
}
function processTranslates(node, options) {
    const { lang, langPriorityList, hashLen } = options;
    return node.properties.forEach((prop) => {
        const keyName = getObjectPropertyValue(prop);
        if (t.isObjectProperty(prop) &&
            t.isObjectExpression(prop.value) &&
            keyName &&
            // don't apply mutation to hashed value
            !keyName.startsWith(HASH_PREFIX)) {
            prop.key = t.identifier(hash(keyName, hashLen));
            // leave only one target language
            const cache = new Map();
            for (const langProp of prop.value.properties) {
                const langPropName = getObjectPropertyValue(langProp);
                if (t.isObjectProperty(langProp) && langPropName) {
                    cache.set(langPropName, langProp);
                }
            }
            let langProp;
            for (const lang of langPriorityList) {
                const currentLangProp = cache.get(lang);
                if (hasTranslatedValue(currentLangProp)) {
                    langProp = currentLangProp;
                    break;
                }
            }
            prop.value.properties = langProp ? [t.objectProperty(t.identifier(lang), langProp.value)] : [];
        }
    });
}

const DEFAULT_LANG = process.env.LANG || 'en';
const DEFAULT_HASH_LEN = 4;
function plugin() {
    return {
        name: '@mova/babel-plugin-i18n',
        visitor: {
            // process.env.LANG -> opts.lang
            MemberExpression(path, state) {
                if (path.get('object').matchesPattern('process.env')) {
                    const node = path.node;
                    if (t.isIdentifier(node.property) && node.property.name === 'LANG') {
                        path.replaceWith(t.stringLiteral(state.opts.lang || DEFAULT_LANG));
                    }
                }
            },
            // store file
            VariableDeclarator(path, state) {
                const lang = state.opts.lang || DEFAULT_LANG;
                const fallbackLangs = state.opts.fallbackLangs || {};
                const hashLen = state.opts.hashLen || DEFAULT_HASH_LEN;
                const langPriorityList = [lang, ...(fallbackLangs[lang] || [])];
                if (state.filename.endsWith('.i18n.ts')) {
                    const node = path.node;
                    if (t.isIdentifier(node.id) && node.id.name === 'translates') {
                        const init = node.init;
                        const processOptions = {
                            lang,
                            langPriorityList,
                            hashLen,
                        };
                        if (t.isCallExpression(init) &&
                            t.isIdentifier(init.callee) &&
                            init.callee.name === 'keyset' &&
                            init.arguments.length &&
                            t.isObjectExpression(init.arguments[0])) {
                            // parse case: `const translates = keyset({ ... });`
                            processTranslates(init.arguments[0], processOptions);
                        }
                        else if (t.isObjectExpression(init)) {
                            // parse case: `const translates = { ... };`
                            processTranslates(init, processOptions);
                        }
                    }
                }
            },
            // source file
            ImportDeclaration(path, state) {
                if (!state.filename.endsWith('.i18n.ts')) {
                    const node = path.node;
                    const importPath = node.source.value;
                    if (importPath.endsWith('.i18n')) {
                        this._movaTransformationEnabled = true;
                    }
                }
            },
            CallExpression(path, state) {
                const hashLen = state.opts.hashLen || 4;
                if (!state.filename.endsWith('.i18n.ts') && this._movaTransformationEnabled) {
                    const node = path.node;
                    if (t.isIdentifier(node.callee) && node.callee.name === 'i18n') {
                        if (node.arguments.length && t.isStringLiteral(node.arguments[0])) {
                            const value = node.arguments[0].value;
                            // don't apply mutation to hashed value
                            if (!value.startsWith(HASH_PREFIX)) {
                                node.arguments[0].value = hash(value, hashLen);
                            }
                        }
                    }
                }
            },
        },
    };
}

export { plugin as default };
