function tokenize(template) {
    const tokens = [];
    let valid = true;
    let prev = 0;
    let open = 0;
    for (let cur = 0; cur < template.length; cur++) {
        if (template[cur] === '{') {
            open += 1;
            if (open > 1) {
                valid = false;
                break;
            }
            if (prev !== cur) {
                // text token
                tokens.push(template.slice(prev, cur));
            }
            prev = cur + 1;
        }
        else if (template[cur] === '}') {
            open -= 1;
            if (open < 0 || prev === cur) {
                valid = false;
                break;
            }
            // ref token
            tokens.push([template.slice(prev, cur)]);
            prev = cur + 1;
        }
    }
    if (!valid) {
        return [template];
    }
    if (prev < template.length) {
        // last text token
        tokens.push(template.slice(prev, template.length));
    }
    return tokens;
}

function _nativePlural(lang) {
    try {
        const rules = new Intl.PluralRules(lang);
        rules.resolvedOptions();
        return (count) => rules.select(count);
    }
    catch (error) {
        // noop
    }
}
function i18n(lang, plurals) {
    const plural = _nativePlural(lang) || plurals[lang];
    if (!plural) {
        throw new Error('mova-i18n plurals not configured properly');
    }
    return function keyset(translates) {
        const cache = {};
        function translate(key, params) {
            const keyTranslates = translates[key] ? translates[key][lang] : undefined;
            let translate;
            let tokens;
            if (!keyTranslates) {
                // no translations
                return [''];
            }
            if (typeof keyTranslates === 'string') {
                // singular
                translate = keyTranslates;
            }
            else {
                // plural
                let count = 1;
                if (params !== undefined) {
                    if (typeof params._count === 'number') {
                        count = params._count;
                    }
                    else if (typeof params.count === 'number') {
                        count = params.count;
                    }
                }
                if (count === 0 && keyTranslates.hasOwnProperty('zero')) {
                    translate = keyTranslates.zero;
                }
                else {
                    translate = keyTranslates[plural(count)] || keyTranslates.one || '';
                }
            }
            if (cache.hasOwnProperty(translate)) {
                tokens = cache[translate];
            }
            else {
                tokens = tokenize(translate);
                cache[translate] = tokens;
            }
            return tokens.map((token) => {
                if (Array.isArray(token)) {
                    // ref token
                    const ref = token[0];
                    return params !== undefined && params.hasOwnProperty(ref) ? params[ref] : '';
                }
                // text token
                return token;
            });
        }
        function t(key, params, mod) {
            const res = translate(key, params);
            return mod && mod.__raw__ === true ? res : res.join('');
        }
        return t;
    };
}

function keyset(translates) {
    return translates;
}

/**
 * 1 -> one
 * 2 -> other
 */
const forms$3 = ['one', 'other'];
const plural$2 = (n) => forms$3[n === 1 ? 0 : 1];

/**
 * 1 -> one
 * 2 -> few
 * 5 -> many
 */
const forms$2 = ['one', 'few', 'many'];
const plural$1 = (n) => forms$2[n % 10 === 1 && n % 100 !== 11 ? 0 : n % 10 >= 2 && n % 10 <= 4 && (n % 100 < 10 || n % 100 >= 20) ? 1 : 2];

/**
 * 1 -> one
 * 2 -> few
 * 5 -> other
 */
const forms$1 = ['one', 'few', 'other'];
const plural = (n) => forms$1[n === 1 ? 0 : n >= 2 && n <= 4 ? 1 : 2];

/**
 * https://unicode-org.github.io/cldr-staging/charts/37/supplemental/language_plural_rules.html
 */
const plurals = {
    en: plural$2,
    be: plural$1,
    ru: plural$1,
    cs: plural,
};

/**
 * https://unicode-org.github.io/cldr-staging/charts/37/supplemental/language_plural_rules.html
 */
const forms = {
    en: forms$3,
    be: forms$2,
    ru: forms$2,
    cs: forms$1,
};

export { forms, i18n, keyset, plurals };
