import { constants } from 'node:fs';
import { readdir, stat, access, readFile, writeFile } from 'node:fs/promises';
import { join } from 'node:path';
import { parse } from '@babel/parser';
import * as t from '@babel/types';
import traverse from '@babel/traverse';

async function* walkFiles(dir) {
    let files = await readdir(dir);
    for (let file of files) {
        let pathname = join(dir, file);
        let fileStat = await stat(pathname);
        if (fileStat.isDirectory()) {
            yield* walkFiles(pathname);
        } else {
            yield pathname;
        }
    }
}

async function parseCodeFile(filePath) {
    try {
        await access(filePath, constants.R_OK | constants.W_OK);
        const code = await readFile(filePath, 'utf-8');
        return parse(code, {
            sourceType: 'module',
            plugins: ['typescript', 'jsx'],
        });
    } catch (error) {
        return null;
    }
}

function _isDefaultExport(obj) {
    return obj.hasOwnProperty('default');
}
function cjs(obj) {
    return _isDefaultExport(obj) ? obj.default : obj;
}

function ast2obj(node) {
    return node.properties.reduce((memo, prop) => {
        if (t.isObjectProperty(prop) && (t.isIdentifier(prop.key) || t.isStringLiteral(prop.key))) {
            const key = t.isIdentifier(prop.key) ? prop.key.name : prop.key.value;
            if (t.isStringLiteral(prop.value)) {
                memo[key] = prop.value.value;
            } else if (t.isObjectExpression(prop.value)) {
                memo[key] = ast2obj(prop.value);
            }
        }
        return memo;
    }, {});
}
function extractStoreTranslates(ast) {
    let translates = {};
    cjs(traverse)(ast, {
        VariableDeclarator(path) {
            const node = path.node;
            if (t.isIdentifier(node.id) && node.id.name === 'translates') {
                const init = node.init;
                if (
                    t.isCallExpression(init) &&
                    t.isIdentifier(init.callee) &&
                    init.callee.name === 'keyset' &&
                    init.arguments.length &&
                    t.isObjectExpression(init.arguments[0])
                ) {
                    // parse case: `const translates = keyset({ ... });`
                    translates = ast2obj(init.arguments[0]);
                } else if (t.isObjectExpression(init)) {
                    // parse case: `const translates = { ... };`
                    translates = ast2obj(init);
                }
            }
        },
    });
    return translates;
}

function _log(filename, time) {
    const message = [new Date().toISOString(), String(Math.round(time)).padStart(4) + 'ms', filename];
    console.log(message.join(' '));
}
async function _processFile(filename, config) {
    if (config.include && config.include.every((path) => !filename.startsWith(path))) {
        return;
    }
    if (config.exclude && config.exclude.some((path) => filename.startsWith(path))) {
        return;
    }
    if (filename.endsWith('.i18n.ts')) {
        const start = Date.now();
        const ast = await parseCodeFile(filename);
        const translates = extractStoreTranslates(ast);
        _log(filename, Date.now() - start);
        return translates;
    }
}
async function exporter(config) {
    const configText = await readFile(config.prettierConfigPath, 'utf-8');
    const prettierConfig = JSON.parse(configText);
    const internalConfig = {
        ...config,
        prettierConfig: { ...prettierConfig, parser: 'typescript' },
    };
    const dir = config.src;
    // initial processing
    const cache = {};
    function isTranslatesEql(t1, t2) {
        return JSON.stringify(t1) === JSON.stringify(t2);
    }
    for await (const filename of walkFiles(dir)) {
        const translates = await _processFile(filename, internalConfig);
        if (translates) {
            for (const key of Object.keys(translates)) {
                if (!cache[key]) {
                    cache[key] = [];
                }
                const cacheTranslates = cache[key].find((t) => isTranslatesEql(t.t, translates[key]));
                if (cacheTranslates) {
                    cacheTranslates.files.push(filename);
                } else {
                    cache[key].push({
                        files: [filename],
                        t: translates[key],
                    });
                }
            }
        }
    }
    // sort keys
    const keys = Object.keys(cache).sort((a, b) => a.localeCompare(b));
    const movaExport = keys.reduce((memo, key) => {
        const values = cache[key];
        memo[key] = values.length > 1 ? values.sort((a, b) => b.files.length - a.files.length) : values;
        return memo;
    }, {});
    // save mova export file
    await writeFile(join(process.cwd(), 'mova-export.json'), JSON.stringify(movaExport, null, 2));
}

const BASE_DIR = process.cwd();
async function run() {
    const configPath = join(BASE_DIR, 'mova.config.json');
    try {
        await access(configPath, constants.R_OK);
    } catch (error) {
        console.error('no mova.config.json was found');
        process.exit(1);
        return;
    }
    const configText = await readFile(configPath, 'utf-8');
    const config = JSON.parse(configText);
    await exporter(config);
}
run().catch((error) => {
    console.error(error);
    process.exit(1);
});
