import { constants } from 'node:fs';
import { readdir, stat, access, readFile, writeFile, watch } from 'node:fs/promises';
import { join } from 'node:path';
import { createHash } from 'crypto';
import { parse } from '@babel/parser';
import * as t from '@babel/types';
import traverse from '@babel/traverse';
import prettier from 'prettier';
import generate from '@babel/generator';
import { forms } from 'mova-i18n';

async function* walkFiles(dir) {
    let files = await readdir(dir);
    for (let file of files) {
        let pathname = join(dir, file);
        let fileStat = await stat(pathname);
        if (fileStat.isDirectory()) {
            yield* walkFiles(pathname);
        } else {
            yield pathname;
        }
    }
}

function hash(data, len = 8) {
    return createHash('sha1').update(JSON.stringify(data)).digest('hex').slice(0, len);
}
function translateHash(data) {
    return hash({ ...data, _meta: undefined });
}

async function parseCodeFile(filePath) {
    try {
        await access(filePath, constants.R_OK | constants.W_OK);
        const code = await readFile(filePath, 'utf-8');
        return parse(code, {
            sourceType: 'module',
            plugins: ['typescript', 'jsx'],
        });
    } catch (error) {
        return null;
    }
}

function _isDefaultExport(obj) {
    return obj.hasOwnProperty('default');
}
function cjs(obj) {
    return _isDefaultExport(obj) ? obj.default : obj;
}

function ast2obj(node) {
    return node.properties.reduce((memo, prop) => {
        if (t.isObjectProperty(prop) && (t.isIdentifier(prop.key) || t.isStringLiteral(prop.key))) {
            const key = t.isIdentifier(prop.key) ? prop.key.name : prop.key.value;
            if (t.isStringLiteral(prop.value)) {
                memo[key] = prop.value.value;
            } else if (t.isObjectExpression(prop.value)) {
                memo[key] = ast2obj(prop.value);
            }
        }
        return memo;
    }, {});
}
function extractStoreTranslates(ast) {
    let translates = {};
    cjs(traverse)(ast, {
        VariableDeclarator(path) {
            const node = path.node;
            if (t.isIdentifier(node.id) && node.id.name === 'translates') {
                const init = node.init;
                if (
                    t.isCallExpression(init) &&
                    t.isIdentifier(init.callee) &&
                    init.callee.name === 'keyset' &&
                    init.arguments.length &&
                    t.isObjectExpression(init.arguments[0])
                ) {
                    // parse case: `const translates = keyset({ ... });`
                    translates = ast2obj(init.arguments[0]);
                } else if (t.isObjectExpression(init)) {
                    // parse case: `const translates = { ... };`
                    translates = ast2obj(init);
                }
            }
        },
    });
    return translates;
}

async function parseStoreFile(filename, config, logger) {
    if (config.include && config.include.every((path) => !filename.startsWith(path))) {
        return;
    }
    if (config.exclude && config.exclude.some((path) => filename.startsWith(path))) {
        return;
    }
    if (filename.endsWith('.i18n.ts')) {
        const start = Date.now();
        const ast = await parseCodeFile(filename);
        const translates = extractStoreTranslates(ast);
        logger(filename, Date.now() - start);
        return translates;
    }
}

function _log$2(filename, time) {
    const message = [new Date().toISOString(), String(Math.round(time)).padStart(4) + 'ms', filename];
    console.log(message.join(' '));
}
async function exporter(config) {
    const configText = await readFile(config.prettierConfigPath, 'utf-8');
    const prettierConfig = JSON.parse(configText);
    const internalConfig = {
        ...config,
        prettierConfig: { ...prettierConfig, parser: 'typescript' },
    };
    const dir = config.src;
    // initial processing
    const cache = {};
    function isTranslatesEql(t1, t2) {
        return JSON.stringify(t1) === JSON.stringify(t2);
    }
    for await (const filename of walkFiles(dir)) {
        const translates = await parseStoreFile(filename, internalConfig, _log$2);
        if (translates) {
            for (const key of Object.keys(translates)) {
                if (!cache[key]) {
                    cache[key] = [];
                }
                const cacheTranslates = cache[key].find((t) => isTranslatesEql(t.t, translates[key]));
                if (cacheTranslates) {
                    cacheTranslates.files.push(filename);
                } else {
                    cache[key].push({
                        hash: translateHash(translates[key]),
                        files: [filename],
                        t: translates[key],
                    });
                }
            }
        }
    }
    // sort keys
    const keys = Object.keys(cache).sort((a, b) => a.localeCompare(b));
    const movaExport = keys.reduce((memo, key) => {
        const values = cache[key].map((item) => {
            if (item.files.length > 1) {
                return {
                    hash: item.hash,
                    files: item.files.sort((a, b) => a.localeCompare(b)),
                    t: item.t,
                };
            }
            return item;
        });
        memo[key] = values.length > 1 ? values.sort((a, b) => b.files.length - a.files.length) : values;
        return memo;
    }, {});
    if (config.hooks && config.hooks.afterExport) {
        await config.hooks.afterExport(movaExport);
    } else {
        await writeFile(join(process.cwd(), 'mova-export.json'), JSON.stringify(movaExport, null, 2));
    }
}

function defaultStoreTemplate(translates) {
    return `
import { i18n as i18nBuilder, keyset, MovaLang, plurals } from 'mova-i18n';

const translates = keyset(${JSON.stringify(translates, null, 4)});

export const i18n = i18nBuilder(process.env.LANG as MovaLang, plurals)(translates);
`.trim();
}
function storeTemplate(nextTranslates) {
    const translates = nextTranslates
        .sort((a, b) => {
            return a[0].localeCompare(b[0]);
        })
        .reduce((memo, [key, value]) => {
            memo[key] = value;
            return memo;
        }, {});
    return defaultStoreTemplate(translates);
}

const { format } = prettier;
async function writeCodeFile(filePath, code, prettierConfig) {
    await writeFile(filePath, format(code, prettierConfig), 'utf-8');
}

function _log$1(filename, time) {
    const message = [new Date().toISOString(), String(Math.round(time)).padStart(4) + 'ms', filename];
    console.log(message.join(' '));
}
async function importer(config) {
    const configText = await readFile(config.prettierConfigPath, 'utf-8');
    const prettierConfig = JSON.parse(configText);
    const internalConfig = {
        ...config,
        prettierConfig: { ...prettierConfig, parser: 'typescript' },
    };
    const dir = config.src;
    let cache = {};
    if (config.hooks && config.hooks.beforeImport) {
        cache = await config.hooks.beforeImport(config.langs);
    } else {
        const movaExportPayload = await readFile(join(process.cwd(), 'mova-export.json'), 'utf-8');
        cache = JSON.parse(movaExportPayload);
    }
    // initial processing
    for await (const filename of walkFiles(dir)) {
        const translates = await parseStoreFile(filename, internalConfig, _log$1);
        if (translates) {
            const nextTranslates = [];
            let isModified = false;
            for (const key of Object.keys(translates)) {
                const localTranslates = translates[key];
                if (cache.hasOwnProperty(key)) {
                    // cache hit
                    const storeHash = translateHash(localTranslates);
                    const cacheTranslates = cache[key].find(({ hash }) => hash === storeHash);
                    if (!cacheTranslates) {
                        console.log(
                            `[ERROR] ${filename} "${key}"`,
                            '\nlocal:',
                            localTranslates,
                            '\nremote:',
                            cache[key],
                        );
                        nextTranslates.push([key, localTranslates]);
                    } else {
                        isModified = true;
                        const meta =
                            localTranslates._meta || cacheTranslates.t._meta
                                ? {
                                      ...localTranslates._meta,
                                      ...cacheTranslates.t._meta,
                                  }
                                : undefined;
                        nextTranslates.push([
                            key,
                            {
                                ...cacheTranslates.t,
                                _meta: meta,
                            },
                        ]);
                    }
                } else {
                    // cache miss
                    nextTranslates.push([key, localTranslates]);
                }
            }
            if (isModified) {
                const content = storeTemplate(nextTranslates);
                await writeCodeFile(filename, content, internalConfig.prettierConfig);
            }
        }
    }
}

function extractSourceTranslates(ast) {
    let enabled = false;
    const i18nCallExpressionList = [];
    const translates = new Set();
    cjs(traverse)(ast, {
        ImportDeclaration(path) {
            const node = path.node;
            const importPath = node.source.value;
            if (importPath.endsWith('.i18n')) {
                enabled = true;
            }
        },
        CallExpression(path) {
            const node = path.node;
            if (t.isIdentifier(node.callee) && node.callee.name === 'i18n') {
                if (node.arguments.length && t.isStringLiteral(node.arguments[0])) {
                    translates.add(node.arguments[0].value);
                    i18nCallExpressionList.push(node);
                }
            }
        },
    });
    return {
        enabled,
        translates,
        i18nCallExpressionList,
    };
}

function generateCode(ast) {
    return cjs(generate)(ast, { retainLines: true, jsonCompatibleStrings: false }).code;
}

function _generatePluralValue(text, lang) {
    if (forms[lang]) {
        return forms[lang].reduce((memo, name) => {
            memo[name] = text;
            return memo;
        }, {});
    }
}
function generateTranslateValue(text, config) {
    const isPlural = text.includes('{count}');
    return config.langs.reduce((memo, lang) => {
        const value = lang === config.lang ? text : '';
        memo[lang] = isPlural ? _generatePluralValue(value, lang) : value;
        return memo;
    }, {});
}

async function processSourceFile(sourceFilePath, config) {
    const storeFilePath = sourceFilePath.replace(/\.tsx?$/, '.i18n.ts');
    const [sourceAst, storeAst] = await Promise.all([parseCodeFile(sourceFilePath), parseCodeFile(storeFilePath)]);
    const { enabled, translates: sourceTranslates, i18nCallExpressionList } = extractSourceTranslates(sourceAst);
    let sourceModified = false;
    let storeModified = false;
    if (enabled) {
        const storeTranslates = extractStoreTranslates(storeAst);
        const nextTranslates = [];
        for (const key of Object.keys(storeTranslates)) {
            if (!sourceTranslates.has(key)) {
                // remove no more actual keys from store
                storeModified = true;
                continue;
            }
            const value = storeTranslates[key];
            const baseValue = value[config.lang];
            if (baseValue) {
                const keyBaseValue = typeof baseValue === 'string' ? baseValue : baseValue.one;
                if (keyBaseValue !== key) {
                    // replace keys in store and source code
                    nextTranslates.push([keyBaseValue, value]);
                    for (const node of i18nCallExpressionList) {
                        if (t.isStringLiteral(node.arguments[0]) && node.arguments[0].value === key) {
                            node.arguments[0].value = keyBaseValue;
                        }
                    }
                    sourceModified = true;
                    storeModified = true;
                } else {
                    nextTranslates.push([key, value]);
                }
            }
        }
        // add new translates from code to store
        for (const key of sourceTranslates.values()) {
            if (!storeTranslates.hasOwnProperty(key)) {
                nextTranslates.push([key, generateTranslateValue(key, config)]);
                storeModified = true;
            }
        }
        // apply modifications
        if (sourceModified) {
            await writeCodeFile(sourceFilePath, generateCode(sourceAst), config.prettierConfig);
        }
        if (storeModified) {
            const content = storeTemplate(nextTranslates);
            await writeCodeFile(storeFilePath, content, config.prettierConfig);
        }
    }
    return { sourceModified, storeModified };
}

function _log(event, filename, time, processed) {
    const message = [
        new Date().toISOString(),
        ('[' + event + ']').padEnd(8),
        processed ? '↻' : '·',
        String(Math.round(time)).padStart(4) + 'ms',
        filename,
    ];
    console.log(message.join(' '));
}
async function findFirstFile(files) {
    for (const file of files) {
        try {
            await access(file, constants.R_OK | constants.W_OK);
            return file;
        } catch (e) {
            // noop
        }
    }
}
async function _processFile(event, filename, config) {
    if (config.include && config.include.every((path) => !filename.startsWith(path))) {
        return;
    }
    if (config.exclude && config.exclude.some((path) => filename.startsWith(path))) {
        return;
    }
    if (filename.endsWith('.ts') || filename.endsWith('.tsx')) {
        const start = Date.now();
        let sourceFilename;
        if (filename.endsWith('.i18n.ts')) {
            sourceFilename = await findFirstFile([
                filename.replace('.i18n.ts', 'tsx'),
                filename.replace('.i18n.ts', 'ts'),
            ]);
        } else {
            sourceFilename = filename;
        }
        if (sourceFilename) {
            const { sourceModified, storeModified } = await processSourceFile(sourceFilename, config);
            _log(event, filename, Date.now() - start, sourceModified || storeModified);
        }
    }
}
async function watcher(config) {
    const configText = await readFile(config.prettierConfigPath, 'utf-8');
    const prettierConfig = JSON.parse(configText);
    const internalConfig = {
        ...config,
        prettierConfig: { ...prettierConfig, parser: 'typescript' },
    };
    const dir = config.src;
    // initial processing
    for await (const filename of walkFiles(dir)) {
        await _processFile('init', filename, internalConfig);
    }
    // watch and process changes
    const watchOpts = {
        encoding: 'utf-8',
        persistent: true,
        recursive: true,
    };
    for await (const event of watch(dir, watchOpts)) {
        const { eventType } = event;
        const filename = join(dir, event.filename);
        if (eventType !== 'change') {
            continue;
        }
        await _processFile(eventType, filename, internalConfig);
    }
}

const BASE_DIR = process.cwd();
async function parseConfigCode() {
    const configPath = join(BASE_DIR, 'mova.config.js');
    await access(configPath, constants.R_OK);
    const configModule = await import(configPath);
    return configModule && configModule.default ? configModule.default : configModule;
}
async function parseConfigJson() {
    const configPath = join(BASE_DIR, 'mova.config.json');
    await access(configPath, constants.R_OK);
    const configText = await readFile(configPath, 'utf-8');
    return JSON.parse(configText);
}
async function parseConfig() {
    try {
        return await parseConfigJson();
    } catch (e) {
        return parseConfigCode();
    }
}
async function run() {
    let config;
    try {
        config = await parseConfig();
    } catch (error) {
        console.error('no mova.config.json was found');
        process.exit(1);
        return;
    }
    const command = process.argv[2];
    if (command === 'export') {
        await exporter(config);
    } else if (command === 'import') {
        await importer(config);
    } else {
        await watcher(config);
    }
}
run().catch((error) => {
    console.error(error);
    process.exit(1);
});
