'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var node_path = require('node:path');
var node_fs = require('node:fs');
var promises = require('node:fs/promises');
var parser = require('@babel/parser');
var t = require('@babel/types');
var traverse = require('@babel/traverse');
var prettier = require('prettier');
var generate = require('@babel/generator');
var movaI18n = require('mova-i18n');

function _interopDefaultLegacy(e) {
    return e && typeof e === 'object' && 'default' in e ? e : { default: e };
}

function _interopNamespace(e) {
    if (e && e.__esModule) return e;
    var n = Object.create(null);
    if (e) {
        Object.keys(e).forEach(function (k) {
            if (k !== 'default') {
                var d = Object.getOwnPropertyDescriptor(e, k);
                Object.defineProperty(
                    n,
                    k,
                    d.get
                        ? d
                        : {
                              enumerable: true,
                              get: function () {
                                  return e[k];
                              },
                          },
                );
            }
        });
    }
    n['default'] = e;
    return Object.freeze(n);
}

var t__namespace = /*#__PURE__*/ _interopNamespace(t);
var traverse__default = /*#__PURE__*/ _interopDefaultLegacy(traverse);
var prettier__default = /*#__PURE__*/ _interopDefaultLegacy(prettier);
var generate__default = /*#__PURE__*/ _interopDefaultLegacy(generate);

async function parseCodeFile(filePath) {
    try {
        await promises.access(filePath, node_fs.constants.R_OK | node_fs.constants.W_OK);
        const code = await promises.readFile(filePath, 'utf-8');
        return parser.parse(code, {
            sourceType: 'module',
            plugins: ['typescript', 'jsx'],
        });
    } catch (error) {
        return null;
    }
}

function _isDefaultExport(obj) {
    return obj.hasOwnProperty('default');
}
function cjs(obj) {
    return _isDefaultExport(obj) ? obj.default : obj;
}

function extractSourceTranslates(ast) {
    let enabled = false;
    const i18nCallExpressionList = [];
    const translates = new Set();
    cjs(traverse__default['default'])(ast, {
        ImportDeclaration(path) {
            const node = path.node;
            const importPath = node.source.value;
            if (importPath.endsWith('.i18n')) {
                enabled = true;
            }
        },
        CallExpression(path) {
            const node = path.node;
            if (t__namespace.isIdentifier(node.callee) && node.callee.name === 'i18n') {
                if (node.arguments.length && t__namespace.isStringLiteral(node.arguments[0])) {
                    translates.add(node.arguments[0].value);
                    i18nCallExpressionList.push(node);
                }
            }
        },
    });
    return {
        enabled,
        translates,
        i18nCallExpressionList,
    };
}

function ast2obj(node) {
    return node.properties.reduce((memo, prop) => {
        if (
            t__namespace.isObjectProperty(prop) &&
            (t__namespace.isIdentifier(prop.key) || t__namespace.isStringLiteral(prop.key))
        ) {
            const key = t__namespace.isIdentifier(prop.key) ? prop.key.name : prop.key.value;
            if (t__namespace.isStringLiteral(prop.value)) {
                memo[key] = prop.value.value;
            } else if (t__namespace.isObjectExpression(prop.value)) {
                memo[key] = ast2obj(prop.value);
            }
        }
        return memo;
    }, {});
}
function extractStoreTranslates(ast) {
    let translates = {};
    cjs(traverse__default['default'])(ast, {
        VariableDeclarator(path) {
            const node = path.node;
            if (t__namespace.isIdentifier(node.id) && node.id.name === 'translates') {
                const init = node.init;
                if (
                    t__namespace.isCallExpression(init) &&
                    t__namespace.isIdentifier(init.callee) &&
                    init.callee.name === 'keyset' &&
                    init.arguments.length &&
                    t__namespace.isObjectExpression(init.arguments[0])
                ) {
                    // parse case: `const translates = keyset({ ... });`
                    translates = ast2obj(init.arguments[0]);
                } else if (t__namespace.isObjectExpression(init)) {
                    // parse case: `const translates = { ... };`
                    translates = ast2obj(init);
                }
            }
        },
    });
    return translates;
}

const { format } = prettier__default['default'];
async function writeCodeFile(filePath, code, prettierConfig) {
    await promises.writeFile(filePath, format(code, prettierConfig), 'utf-8');
}

function generateCode(ast) {
    return cjs(generate__default['default'])(ast, { retainLines: true, jsonCompatibleStrings: false }).code;
}

function _generatePluralValue(text, lang) {
    if (movaI18n.forms[lang]) {
        return movaI18n.forms[lang].reduce((memo, name) => {
            memo[name] = text;
            return memo;
        }, {});
    }
}
function generateTranslateValue(text, config) {
    const isPlural = text.includes('{count}');
    return config.langs.reduce((memo, lang) => {
        const value = lang === config.lang ? text : '';
        memo[lang] = isPlural ? _generatePluralValue(value, lang) : value;
        return memo;
    }, {});
}

function defaultStoreTemplate(translates) {
    return `
import { i18n as i18nBuilder, keyset, MovaLang, plurals } from 'mova-i18n';

const translates = keyset(${JSON.stringify(translates, null, 4)});

export const i18n = i18nBuilder(process.env.LANG as MovaLang, plurals)(translates);
`.trim();
}
function storeTemplate(nextTranslates) {
    const translates = nextTranslates
        .sort((a, b) => {
            return a[0].localeCompare(b[0]);
        })
        .reduce((memo, [key, value]) => {
            memo[key] = value;
            return memo;
        }, {});
    return defaultStoreTemplate(translates);
}

async function processSourceFile(sourceFilePath, config) {
    const storeFilePath = sourceFilePath.replace(/\.tsx?$/, '.i18n.ts');
    const [sourceAst, storeAst] = await Promise.all([parseCodeFile(sourceFilePath), parseCodeFile(storeFilePath)]);
    const { enabled, translates: sourceTranslates, i18nCallExpressionList } = extractSourceTranslates(sourceAst);
    let sourceModified = false;
    let storeModified = false;
    if (enabled) {
        const storeTranslates = extractStoreTranslates(storeAst);
        const nextTranslates = [];
        for (const key of Object.keys(storeTranslates)) {
            if (!sourceTranslates.has(key)) {
                // remove no more actual keys from store
                storeModified = true;
                continue;
            }
            const value = storeTranslates[key];
            const baseValue = value[config.lang];
            if (baseValue) {
                const keyBaseValue = typeof baseValue === 'string' ? baseValue : baseValue.one;
                if (keyBaseValue !== key) {
                    // replace keys in store and source code
                    nextTranslates.push([keyBaseValue, value]);
                    for (const node of i18nCallExpressionList) {
                        if (t__namespace.isStringLiteral(node.arguments[0]) && node.arguments[0].value === key) {
                            node.arguments[0].value = keyBaseValue;
                        }
                    }
                    sourceModified = true;
                    storeModified = true;
                } else {
                    nextTranslates.push([key, value]);
                }
            }
        }
        // add new translates from code to store
        for (const key of sourceTranslates.values()) {
            if (!storeTranslates.hasOwnProperty(key)) {
                nextTranslates.push([key, generateTranslateValue(key, config)]);
                storeModified = true;
            }
        }
        // apply modifications
        if (sourceModified) {
            await writeCodeFile(sourceFilePath, generateCode(sourceAst), config.prettierConfig);
        }
        if (storeModified) {
            const content = storeTemplate(nextTranslates);
            await writeCodeFile(storeFilePath, content, config.prettierConfig);
        }
    }
    return { sourceModified, storeModified };
}

async function* walkFiles(dir) {
    let files = await promises.readdir(dir);
    for (let file of files) {
        let pathname = node_path.join(dir, file);
        let fileStat = await promises.stat(pathname);
        if (fileStat.isDirectory()) {
            yield* walkFiles(pathname);
        } else {
            yield pathname;
        }
    }
}

function _log(event, filename, time, processed) {
    const message = [
        new Date().toISOString(),
        ('[' + event + ']').padEnd(8),
        processed ? '↻' : '·',
        String(Math.round(time)).padStart(4) + 'ms',
        filename,
    ];
    console.log(message.join(' '));
}
async function findFirstFile(files) {
    for (const file of files) {
        try {
            await promises.access(file, node_fs.constants.R_OK | node_fs.constants.W_OK);
            return file;
        } catch (e) {
            // noop
        }
    }
}
async function _processFile(event, filename, config) {
    if (config.include && config.include.every((path) => !filename.startsWith(path))) {
        return;
    }
    if (config.exclude && config.exclude.some((path) => filename.startsWith(path))) {
        return;
    }
    if (filename.endsWith('.ts') || filename.endsWith('.tsx')) {
        const start = Date.now();
        let sourceFilename;
        if (filename.endsWith('.i18n.ts')) {
            sourceFilename = await findFirstFile([
                filename.replace('.i18n.ts', 'tsx'),
                filename.replace('.i18n.ts', 'ts'),
            ]);
        } else {
            sourceFilename = filename;
        }
        if (sourceFilename) {
            const { sourceModified, storeModified } = await processSourceFile(sourceFilename, config);
            _log(event, filename, Date.now() - start, sourceModified || storeModified);
        }
    }
}
async function watcher(config) {
    const configText = await promises.readFile(config.prettierConfigPath, 'utf-8');
    const prettierConfig = JSON.parse(configText);
    const internalConfig = {
        ...config,
        prettierConfig: { ...prettierConfig, parser: 'typescript' },
    };
    const dir = config.src;
    // initial processing
    for await (const filename of walkFiles(dir)) {
        await _processFile('init', filename, internalConfig);
    }
    // watch and process changes
    const watchOpts = {
        encoding: 'utf-8',
        persistent: true,
        recursive: true,
    };
    for await (const event of promises.watch(dir, watchOpts)) {
        const { eventType } = event;
        const filename = node_path.join(dir, event.filename);
        if (eventType !== 'change') {
            continue;
        }
        await _processFile(eventType, filename, internalConfig);
    }
}

exports.watcher = watcher;
