import { readFile } from 'node:fs/promises';
import { join } from 'node:path';
import type { MovaWatcherConfig, MovaWatcherTranslatesStore, MovaWatcherInternalConfig } from 'types';
import { walkFiles } from 'lib/walkFiles';
import { MovaTranslateValue } from 'types';
import { storeTemplate } from 'lib/storeTemplate';
import { writeCodeFile } from 'lib/writeCodeFile';
import { translateHash } from 'lib/translateHash';
import { parseStoreFile } from 'lib/parseStoreFile';

function _log(filename: string, time: number): void {
    const message = [new Date().toISOString(), String(Math.round(time)).padStart(4) + 'ms', filename];

    console.log(message.join(' '));
}

export async function importer(config: MovaWatcherConfig): Promise<void> {
    const configText = await readFile(config.prettierConfigPath, 'utf-8');
    const prettierConfig = JSON.parse(configText);

    const internalConfig: MovaWatcherInternalConfig = {
        ...config,
        prettierConfig: { ...prettierConfig, parser: 'typescript' },
    };

    const dir = config.src;

    let cache: MovaWatcherTranslatesStore = {};

    if (config.hooks && config.hooks.beforeImport) {
        cache = await config.hooks.beforeImport(config.langs);
    } else {
        const movaExportPayload = await readFile(join(process.cwd(), 'mova-export.json'), 'utf-8');
        cache = JSON.parse(movaExportPayload);
    }

    // initial processing
    for await (const filename of walkFiles(dir)) {
        const translates = await parseStoreFile(filename, internalConfig, _log);

        if (translates) {
            const nextTranslates: Array<[string, MovaTranslateValue]> = [];
            let isModified = false;

            for (const key of Object.keys(translates)) {
                const localTranslates = translates[key];

                if (cache.hasOwnProperty(key)) {
                    // cache hit
                    const storeHash = translateHash(localTranslates);
                    const cacheTranslates = cache[key].find(({ hash }) => hash === storeHash);

                    if (!cacheTranslates) {
                        console.log(
                            `[ERROR] ${filename} "${key}"`,
                            '\nlocal:',
                            localTranslates,
                            '\nremote:',
                            cache[key],
                        );

                        nextTranslates.push([key, localTranslates]);
                    } else {
                        isModified = true;

                        const meta =
                            localTranslates._meta || cacheTranslates.t._meta
                                ? {
                                      ...localTranslates._meta,
                                      ...cacheTranslates.t._meta,
                                  }
                                : undefined;

                        nextTranslates.push([
                            key,
                            {
                                ...cacheTranslates.t,
                                _meta: meta,
                            },
                        ]);
                    }
                } else {
                    // cache miss
                    nextTranslates.push([key, localTranslates]);
                }
            }

            if (isModified) {
                const content = storeTemplate(nextTranslates);

                await writeCodeFile(filename, content, internalConfig.prettierConfig);
            }
        }
    }
}
