const path = require('path');
const webpack = require('webpack');
const HtmlWebpackPlugin = require('html-webpack-plugin');
const MiniCssExtractPlugin = require('mini-css-extract-plugin');
const { lang } = require('../consts');

const rootDir = path.resolve(__dirname, '..', '..');

function pages(service, cdn) {
    return [
        new HtmlWebpackPlugin({
            publicPath: cdn ? cdn : '/',
            template: path.resolve(rootDir, `./src/${service}.html`),
            filename: `pages/${service + (cdn ? '-cdn' : '')}/${lang}/index.html`,
            excludeChunks: ['app403', 'app502'],
            templateParameters: {
                cdn: JSON.stringify(cdn || false),
            },
        }),
        new HtmlWebpackPlugin({
            title: '403',
            publicPath: cdn ? cdn : '/',
            template: path.resolve(rootDir, `./src/${service}.html`),
            filename: `pages/${service + (cdn ? '-cdn' : '')}/${lang}/index_403.html`,
            chunks: ['app403'],
            templateParameters: {
                cdn: JSON.stringify(cdn || false),
            },
        }),
        new HtmlWebpackPlugin({
            title: '502',
            publicPath: cdn ? cdn : '/',
            template: path.resolve(rootDir, `./src/${service}.html`),
            filename: `pages/${service + (cdn ? '-cdn' : '')}/${lang}/index_502.html`,
            chunks: ['app502'],
            templateParameters: {
                cdn: JSON.stringify(cdn || false),
            },
        }),
    ];
}

module.exports = (mode) => {
    return {
        mode,
        entry: {
            app: path.resolve(rootDir, 'src/app/app.ts'),
            app403: path.resolve(rootDir, 'src/app/app403.ts'),
            app502: path.resolve(rootDir, 'src/app/app502.ts'),
        },
        output: {
            publicPath: '/',
            path: path.resolve(rootDir, 'dist/web'),
            chunkFilename: `static/${lang}/[name].[contenthash].js`,
            filename: `static/${lang}/[name].[contenthash].js`,
        },
        resolve: {
            modules: [path.resolve(rootDir, './src'), 'node_modules'],
            extensions: ['.ts', '.tsx', '.js'],
            alias: {
                xlsx: 'xlsx/dist/xlsx.mini.min.js',
            },
        },
        plugins: [
            new MiniCssExtractPlugin({
                filename: `static/${lang}/[name].[contenthash].css`,
                chunkFilename: `static/${lang}/[name].[contenthash].css`,
                ignoreOrder: true,
            }),
            // see nonce-sha256: drivematics-server/src/lib/csp/getVlootkitCSP.ts
            ...pages('drivematics'),
            ...pages('drivematics', 'https://yastatic.net/s3/drivematics/'),
            // see nonce-sha256: drivematics-server/src/lib/csp/getVlootkitCSP.ts
            ...pages('vlootkit'),
            ...pages('vlootkit', 'https://vlootkit-frontend.s3.eu-central-1.amazonaws.com/'),
            new webpack.DefinePlugin({
                'process.env.DM_VERSION': JSON.stringify(process.env.DM_VERSION),
                'process.env.DM_BACKEND_CLUSTER': JSON.stringify(process.env.DM_BACKEND_CLUSTER),
                'process.env.DM_CACHE_REQUEST': JSON.stringify(false),
            }),
        ],
        module: {
            rules: [
                {
                    test: /\.tsx?$/,
                    exclude: /node_modules/,
                    use: [
                        {
                            loader: 'babel-loader',
                            options: {
                                cacheDirectory: `./node_modules/.cache/babel-loader/${lang}/`,
                                presets: [
                                    ['@babel/preset-env', { loose: true }],
                                    '@babel/preset-react',
                                    '@babel/preset-typescript',
                                ],
                                plugins: [
                                    [
                                        require.resolve('mova-babel-plugin-i18n'),
                                        { lang, fallbackLangs: { ru: ['en'], cs: ['en'] } },
                                    ],
                                ],
                                // plugins: ['@babel/plugin-transform-runtime', [require.resolve('mova-babel-plugin-i18n'), {lang, fallbackLangs: {ru: ['en']}}]],
                            },
                        },
                    ],
                },
                {
                    test: /\.svg$/,
                    oneOf: [
                        {
                            test: /\.(component|inline)\.svg$/,
                            exclude: /node_modules/,
                            use: ['@svgr/webpack'],
                        },
                        {
                            test: /images(.)+\.svg$/,
                            type: 'asset',
                            generator: {
                                filename: `static/${lang}/f/[contenthash][ext]`,
                            },
                        },
                    ],
                },
                {
                    test: /\.(png|jpg|gif)$/,
                    type: 'asset',
                    generator: {
                        filename: `static/${lang}/f/[contenthash][ext]`,
                    },
                },
                {
                    test: /\.css$/,
                    use: [
                        {
                            loader: MiniCssExtractPlugin.loader,
                            options: {
                                esModule: false,
                            },
                        },
                        {
                            loader: 'css-loader',
                            options: {
                                modules: {
                                    localIdentName: '[folder]__[local]__[hash:base64:5]',

                                    mode: (path) => {
                                        if (/global.css$/i.test(path)) {
                                            return 'global';
                                        }

                                        return 'local';
                                    },
                                },
                                importLoaders: 1,
                                esModule: false,
                            },
                        },
                        {
                            loader: 'postcss-loader',
                            options: {
                                postcssOptions: {
                                    config: path.resolve(rootDir),
                                },
                            },
                        },
                    ],
                },
            ],
        },
        optimization: {
            moduleIds: 'deterministic',
            runtimeChunk: 'single',
            splitChunks: {
                minSize: 10240,
                cacheGroups: {
                    vendor: {
                        test: /[\\/]node_modules[\\/]/,
                        name: 'vendors',
                        chunks: 'all',
                    },
                },
            },
            emitOnErrors: false,
        },
        stats: {
            assetsSort: '!size',
            colors: true,
            logging: 'error',
            reasons: false,
            children: false,
            usedExports: false,
        },
    };
};
