import { isValidJSONString } from 'utils/isValidJSONString';

import { HttpStatusCode } from 'shared/consts/HttpStatusCode';
import { ErrorLoggerSource, logError } from 'shared/helpers/errorLogger/errorLogger';
import { isVlootkit } from 'shared/helpers/isVlootkit/isVlootkit';

const random = 200000000000;
export const ABORT_ERROR_KEY = 'AbortError';

export enum RequestHelperMethod {
    GET = 'get',
    POST = 'post',
    PATCH = 'PATCH',
    DELETE = 'DELETE',
}

export enum RequestApiPrefix {
    DEFAULT = 'api/leasing',
    PASSPORT = 'passport',
    MOCK = 'mock',
}

export enum specialQueryParamsKeys {
    FORCE = 'force',
    FORCE_ABSENT_KEY = 'FORCE_ABSENT_KEY',
}

export interface RequestHelperConfig {
    api: string;

    method?: RequestHelperMethod;
    cgiObj?: {
        [key: string]: any;
    };
    apiPrefix?: RequestApiPrefix;
    timeout?: number;
}

export interface RequestHelperConfigs {
    [key: string]: RequestHelperConfig;
}

export interface RequestOptions {
    queryParams?: any;
    specialQueryParams?: any;
    body?: any;
}

export interface RequestError {
    status: number;
    statusText: string;
    error_details: {
        debug_message: string;
        http_code: number;
        ui_message: string;
        special_info: any;
    };
}

function _fetch(input: string, init?: RequestInit, retry = 1): Promise<Response> {
    return fetch(input, init).then((response) => {
        if (response.status === HttpStatusCode.UNAUTHORIZED && isVlootkit() && retry > 0) {
            return fetch('/auth/api/refresh', {
                method: 'GET',
                credentials: 'same-origin',
            }).then(() => _fetch(input, init, retry - 1));
        }

        return response;
    });
}

export class RequestHelper {
    method: RequestHelperMethod = RequestHelperMethod.GET;
    requestConfigs: RequestHelperConfigs = {} as RequestHelperConfigs;

    controller = {} as AbortController;

    constructor(props: { requestConfigs: RequestHelperConfigs }) {
        this.updateController();
        this.requestConfigs = props.requestConfigs;
    }

    matchQueryParams(str: string, options: any): string {
        options &&
            options.queryParams &&
            Object.entries(options.queryParams).map((param: any[]) => {
                let reqExp = new RegExp('\\${' + param[0] + '}', 'ig');
                str = str.replace(reqExp, param[1]);
            });

        return str;
    }

    matchCgiParams(cgiObj: any, options): string {
        let queryParams: any = options && options.queryParams;
        let specialQueryParams: any = options && options.specialQueryParams;

        let cgi = Object.entries(cgiObj).reduce((_p: any, _c: any) => {
            if (_c[1]) {
                _p.push(`${_c[0]}=${_c[1]}`);
            } else if (_c[1] === null) {
                if (queryParams && queryParams.hasOwnProperty(_c[0]) && queryParams[_c[0]]) {
                    _p.push(`${_c[0]}=${queryParams[_c[0]]}`);
                }
                if (specialQueryParams && specialQueryParams.hasOwnProperty(_c[0]) && specialQueryParams[_c[0]]) {
                    let queryObj = specialQueryParams[_c[0]];
                    if (queryObj.key === specialQueryParamsKeys.FORCE) {
                        _p.push(`${_c[0]}=${queryObj.value}`);
                    }
                }
            }

            return _p;
        }, []);
        let extraForcedCgi: string[] = specialQueryParams
            ? Object.entries(specialQueryParams)?.reduce((acc: string[], entry: any) => {
                  if (entry[1]?.key === specialQueryParamsKeys.FORCE_ABSENT_KEY) {
                      acc.push(`${entry[0]}=${entry[1]?.value}`);
                  }

                  return acc;
              }, [])
            : [];

        return cgi.concat(extraForcedCgi).join('&');
    }

    headers() {
        return {
            'Content-Type': 'application/json',
            admbuild: process.env.DM_VERSION?.replace(/[.-]/gi, '') || '',
        };
    }

    addCGI(props, options): string {
        let matchParams = (props.cgiObj && this.matchCgiParams(props.cgiObj, options)) || '';
        let url = new URLSearchParams(matchParams);
        url.append(`reqid`, `${new Date().getTime()}-${(Math.random() * random).toFixed(0)}-drivematics`);
        url.append('lang', process.env.LANG || 'en');

        return url.toString();
    }

    exec(request: string, options?: RequestOptions) {
        this.updateController();
        if (this.requestConfigs && this.requestConfigs.hasOwnProperty(request)) {
            let props = this.requestConfigs[request];
            let body = options?.body || null;
            let pref = props.apiPrefix || RequestApiPrefix.DEFAULT || '';
            let method = props.method || this.method;

            let endpoint = `${pref}/${this.matchQueryParams(props.api, options)}`;

            let cgi = this.addCGI(props, options);

            let url = endpoint + `?${cgi}`;

            if (!url.startsWith('https://') && !url.startsWith('//') && !url.startsWith('/')) {
                // fix relative path request
                url = '/' + url;
            }

            return _fetch(url, {
                signal: this.controller?.signal,
                method,
                headers: this.headers(),
                referrerPolicy: 'origin-when-cross-origin',
                credentials: 'include',
                ...(body && { body: JSON.stringify(body) }),
            })
                .then(async (response) => {
                    if (!response.ok) {
                        return Promise.reject(response);
                    } else {
                        let responseText = await response.text();
                        if (isValidJSONString(responseText)) {
                            return JSON.parse(responseText);
                        } else {
                            return responseText;
                        }
                    }
                })
                .catch(async (response) => {
                    if (response.name === ABORT_ERROR_KEY) {
                        return Promise.resolve({ meta: ABORT_ERROR_KEY });
                    }

                    let errorData = await response.text();
                    try {
                        errorData = JSON.parse(errorData);
                    } catch (err) {}

                    logError(
                        ErrorLoggerSource.NETWORK,
                        new Error(
                            JSON.stringify({
                                endpoint,
                                status: response.status,
                            }),
                        ),
                        { fullError: errorData },
                    );

                    return Promise.reject({
                        status: response.status,
                        statusText: response.statusText,
                        error: errorData,
                    });
                });
        } else {
            return Promise.reject(new Error(`Don't have request handler ${request}`));
        }
    }

    updateController() {
        try {
            this.controller = window?.AbortController ? new window.AbortController() : ({} as AbortController);
        } catch (e) {
            this.controller = {} as AbortController;
        }
    }

    abort() {
        this.controller?.abort?.();
    }
}
