import React, { MutableRefObject, useContext, useRef } from 'react';

import { ONE_SECOND } from 'constants/constants';

import { getBeaconSearchMode } from 'utils/getBeaconSearchMode';

import {
    BEACON_REQUESTS,
    REQUESTS,
} from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/BeaconSwitchMode/request';
import {
    BEACON_SEARCH_MODE,
    BEACON_SEARCH_STATE,
} from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/BeaconSwitchMode/types';
import NotificationCenterContext, { addNotification } from 'components/NotificationCenter/store';
import { NotificationIconType } from 'components/NotificationCenter/types';
import { Switcher } from 'components/ui/Switcher';
import Main2 from 'components/ui/Text/Main2';
import MainText3 from 'components/ui/Text/MainText3';

import { RequestHelper } from '../../../../../../../request-helper/src';

import { i18n } from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/BeaconSwitchMode/index.i18n';

import styles from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/BeaconSwitchMode/index.css';

interface IBeaconSwitchModeProps {
    car_id: string;
    imei: string;
    operation_mode: number | string;
    beacon_search_state: number | string;
}

export const BeaconSwitchMode = (props: IBeaconSwitchModeProps) => {
    let [state, setState] = React.useState(props.beacon_search_state == BEACON_SEARCH_STATE.ENABLED);
    let operationModeIsEnabled = props.operation_mode == BEACON_SEARCH_MODE.ENABLED;
    let isSwitchBeaconSearchModeEnabled = getBeaconSearchMode();

    const request: MutableRefObject<RequestHelper> = useRef(
        new RequestHelper({
            requestConfigs: BEACON_REQUESTS,
        }),
    );

    let { notificationDispatch } = useContext(NotificationCenterContext) || {};

    const onChange = (state) => {
        setState(state);

        request.current
            .exec(REQUESTS.SWITCH_BEACON_MODE, {
                queryParams: {
                    search_state: state ? '1' : '0',
                    car_id: props.car_id,
                },

                body: { imeis: [props.imei] },
            })
            .then(() => {
                let notification = {
                    title: state ? i18n('Search mode was enabled') : i18n('Search mode was disabled'),
                };

                notificationDispatch(addNotification(notification));
            })
            .catch(({ error }) => {
                let notification = {
                    title: `Error: ${error.error_details?.http_code}`,
                    iconType: NotificationIconType.ERROR,
                    description: `${error.error_details?.debug_message}`,
                    delay: ONE_SECOND,
                };

                notificationDispatch(addNotification(notification));
                setState(!state);
            });
    };

    return (
        <div className={styles.component}>
            <div className={styles.column}>
                <div className={state ? styles.beacon_mode : ''}>
                    <Main2>{i18n('Search mode')}</Main2>
                </div>
                <div>
                    <MainText3 className={styles.subtitle}>
                        {state
                            ? i18n('The beacon transmits data every 20 minutes')
                            : i18n('The beacon transmits data 2 times a day')}
                    </MainText3>
                </div>
                <div>
                    <MainText3 className={styles.subtitle}>
                        {operationModeIsEnabled !== state ? i18n('The beacon is waiting for the mode switch...') : ''}
                    </MainText3>
                </div>
            </div>
            <div className={styles.column}>
                {isSwitchBeaconSearchModeEnabled && (
                    <Switcher
                        checked={state}
                        onChange={onChange}
                    />
                )}
            </div>
        </div>
    );
};
