import React, { MutableRefObject, useEffect, useRef, useState } from 'react';

import { ICarInfo } from 'utils/car/types';

import { ControlType } from 'entities/Car';
import { CarSchema } from 'entities/Car/types/CarSchema';

import {
    CAR_CONTROL_REQUESTS,
    REQUESTS,
} from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/CarControlConfirm/request';
import ModalCarInfo from 'components/Cars/CarCard/CarCardOverview/ModalCarInfo';
import Confirm from 'components/ui/Confirm';

import { RequestHelper } from '../../../../../../../request-helper/src';

import { i18n } from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/CarControlConfirm/index.i18n';

interface ICarControlConfirmProps {
    title?: string;
    onClose: () => void;
    carInfo: ICarInfo | CarSchema;
    actionType: 'command' | 'add_tag' | 'remove_tag';
    command?: ControlType | null;
    addingTag?: string | null;
    removingTagNames?: string[] | null;
}

export default function CarControlConfirm(props: ICarControlConfirmProps) {
    let { title, onClose, carInfo, actionType } = props;
    const [isLoading, setLoading] = useState<boolean>(false);
    const [error, setError] = useState<Error | null>(null);
    const request: MutableRefObject<RequestHelper> = useRef(
        new RequestHelper({ requestConfigs: CAR_CONTROL_REQUESTS }),
    );

    const onConfirm = () => {
        setLoading(true);
        let carActionRequest: Promise<any> | null = null;
        switch (actionType) {
            case 'command':
                let command = props.command?.toUpperCase();
                carActionRequest = request.current.exec(REQUESTS.CAR_CONTROL, {
                    queryParams: {
                        command,
                        car_id: carInfo.id,
                    },
                });

                break;
            case 'add_tag':
                let { addingTag } = props;
                carActionRequest = request.current.exec(REQUESTS.ADD_CAR_TAG, {
                    body: {
                        tag: addingTag,
                        car_id: carInfo.id,
                    },
                });

                break;
            case 'remove_tag':
                let { removingTagNames } = props;
                carActionRequest = request.current.exec(REQUESTS.REMOVE_CAR_TAG, {
                    body: {
                        tag_names: removingTagNames,
                        object_ids: carInfo.id,
                    },
                });

                break;
        }

        if (carActionRequest) {
            carActionRequest
                .then(() => {
                    onClose();
                })
                .catch((error) => {
                    setError(error);
                })
                .finally(() => {
                    setLoading(false);
                });
        }
    };

    useEffect(() => {
        return () => {
            request.current.abort();
        };
    }, []);

    const actionButtonTitle = isLoading ? '...' : error ? i18n('Try again') : i18n('Yes, execute');

    return (
        <Confirm
            title={title ?? i18n('Execute the command?')}
            description={<ModalCarInfo carInfo={carInfo} />}
            confirmButtonTitle={actionButtonTitle}
            closeButtonTitle={i18n('No')}
            onConfirm={onConfirm}
            onClose={onClose}
            isLoading={isLoading}
            error={error}
        />
    );
}
