import React, { useEffect, useState } from 'react';

import { checkModelForBlock } from 'utils/checkModelForBlock';
import { isManageDoors } from 'utils/isManageDoors';
import { isShowCarService } from 'utils/isShowCarService';

import { ControlType } from 'entities/Car';

import CarControlConfirm from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/CarControlConfirm';
import { CAR_BLOCK_TAGS } from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/CarControlConfirm/types';
import { CarControlButton } from 'components/Cars/CarControlButton';
import { SERVICE_TAG } from 'components/Cars/constants';

import { i18n } from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/CarControls/index.i18n';

import style from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/CarControls/index.css';
import coreStyle from 'components/Content/index.css';

const CarControls = ({ isLoading, carInfo }: { isLoading: boolean; carInfo: any }) => {
    const [isBlockConfirmOpen, setIsBlockConfirmOpen] = useState<boolean>(false);
    const [isServiceConfirmOpen, setIsServiceConfirmOpen] = useState<boolean>(false);
    const [isCarInService, setIsCarInService] = useState<boolean>(false);

    let [command, setCommand] = useState<ControlType | null>(null);

    useEffect(() => {
        if (carInfo) {
            let isCarInService = (carInfo.tags ?? []).find((tag) => tag.tag === SERVICE_TAG) ?? false;
            setIsCarInService(isCarInService);
        }
    }, [carInfo]);

    const onClick = (command) => {
        setCommand(command);
    };

    const openServiceConfirm = () => {
        setIsServiceConfirmOpen(true);
    };

    const closeServiceConfirm = () => {
        setIsServiceConfirmOpen(false);
    };

    let { tags } = carInfo ?? {};

    const isModelForBlock = checkModelForBlock(carInfo?.model_id);
    const isCarInBlockProcess = tags?.some((tag) =>
        [
            CAR_BLOCK_TAGS.telematics_deferred_polite_wireless_block,
            CAR_BLOCK_TAGS.telematics_deferred_wireless_block,
        ].includes(tag.tag),
    );

    const isCarBlocked = tags?.some((tag) => tag.tag === CAR_BLOCK_TAGS.leasing_car_is_blocked);

    return (
        <div className={style.car_controls_container}>
            {isLoading ? (
                <>
                    <div className={`${coreStyle.shimmer} ${style.control_shimmer}`} />
                    <div className={`${coreStyle.shimmer} ${style.control_shimmer}`} />
                </>
            ) : (
                <>
                    {isManageDoors() ? (
                        <div className={style.car_control_container}>
                            <CarControlButton
                                onClick={onClick.bind(null, ControlType.open_doors)}
                                type={ControlType.open_doors}
                            />

                            <CarControlButton
                                onClick={onClick.bind(null, ControlType.close_doors)}
                                type={ControlType.close_doors}
                            />
                        </div>
                    ) : null}
                    {isShowCarService() ? (
                        <CarControlButton
                            onClick={openServiceConfirm}
                            type={isCarInService ? ControlType.return_from_service : ControlType.move_to_service}
                        />
                    ) : null}
                    {isModelForBlock ? (
                        <CarControlButton
                            onClick={setIsBlockConfirmOpen.bind(null, true)}
                            type={isCarInBlockProcess || isCarBlocked ? ControlType.unblock : ControlType.block}
                        />
                    ) : null}
                </>
            )}

            {isBlockConfirmOpen ? (
                <CarControlConfirm
                    carInfo={carInfo}
                    actionType={isCarInBlockProcess ? 'remove_tag' : 'add_tag'}
                    addingTag={
                        isCarBlocked
                            ? CAR_BLOCK_TAGS.telematics_deferred_wireless_unblock
                            : !isCarInBlockProcess
                            ? CAR_BLOCK_TAGS.telematics_deferred_polite_wireless_block
                            : null
                    }
                    removingTagNames={
                        isCarInBlockProcess ? [CAR_BLOCK_TAGS.telematics_deferred_polite_wireless_block] : null
                    }
                    onClose={setIsBlockConfirmOpen.bind(null, false)}
                />
            ) : null}
            {command ? (
                <CarControlConfirm
                    carInfo={carInfo}
                    actionType="command"
                    command={command}
                    onClose={setCommand.bind(null, null)}
                />
            ) : null}
            {isServiceConfirmOpen ? (
                <CarControlConfirm
                    title={!isCarInService ? i18n('Move to service?') : i18n('Return from service?')}
                    carInfo={carInfo}
                    actionType={isCarInService ? 'remove_tag' : 'add_tag'}
                    addingTag={!isCarInService ? SERVICE_TAG : null}
                    removingTagNames={isCarInService ? [SERVICE_TAG] : null}
                    onClose={closeServiceConfirm}
                />
            ) : null}
        </div>
    );
};

export default CarControls;
