import * as React from 'react';

import { EMPTY_DATA } from 'constants/constants';

import { ICarInfo, ICarScoring } from 'utils/car/types';
import { isCarAlarmSystemFlag } from 'utils/isCarAlarmSystemFlag';
import { showCarScoring } from 'utils/showCarScoring';

import { getCarSpeed } from 'entities/Car/helpers/getCarSpeed/getCarSpeed';
import { getCarTelematicSpeed } from 'entities/Car/helpers/getCarTelematicSpeed/getCarTelematicSpeed';

import { getLastUpdateString } from 'shared/helpers/getLastUpdateString/getLastUpdateString';
import { GPSValueIcon } from 'shared/ui/GPSValueIcon/GPSValueIcon';
import { Scoring } from 'shared/ui/Scoring/Scoring';

import GSMSignalIcon, { GSMSignalStrength } from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/GSMSignalIcon';
import CarStatus from 'components/Cars/CarStatus';
import { StatusTypes } from 'components/types';
import { Status } from 'components/ui/Status';

import { i18n } from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/CarTabInformation/TelematicsTab/index.i18n';

import style from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/CarTabInformation/TelematicsTab/index.css';

interface ITelematicsTabProps {
    carInfo: ICarInfo & ICarScoring;
    carTelematics: any;
}

export const getTelematicsTab = (props: ITelematicsTabProps) => {
    let { carTelematics, carInfo } = props;
    let { status = StatusTypes.UNKNOWN, telematics } = carInfo ?? {};
    let speed = getCarSpeed(carInfo) ?? getCarTelematicSpeed(carTelematics);
    let aggressiveRank = carInfo?.aggressive_rank;
    let fuel_level = telematics?.fuel_level ?? telematics?.fuel_level ?? null;
    let fuel_distance = telematics?.fuel_distance ?? telematics?.fuel_distance ?? null;
    let ext_voltage = carTelematics?.ext_voltage ?? telematics?.ext_voltage ?? null;
    let engine_temperature =
        Number(carTelematics?.engine_temperature ?? telematics?.engine_temperature).toFixed(1) ?? null;
    let mileage = carTelematics?.mileage ?? telematics?.mileage ?? null;
    let gsm_signal_level =
        status === StatusTypes.NO_SIGNAL
            ? GSMSignalStrength.ZERO
            : carTelematics?.gsm_signal_level ?? telematics?.gsm_signal_level ?? null;
    const GPSInview =
        status === StatusTypes.NO_SIGNAL ? 0 : carTelematics?.VEGA_GPS_INVIEW ?? telematics?.VEGA_GPS_INVIEW ?? null;
    const GPSUsed = status === StatusTypes.NO_SIGNAL ? 0 : carTelematics?.gps_used ?? telematics?.gps_used ?? null;
    let lastUpdate = carTelematics?.gsm_signal_level_updated_at;
    const alarmSystem = carTelematics && Boolean(carTelematics['2856']);
    const alert = carTelematics && Boolean(carTelematics['2857']);
    const alertText = alert ? (
        <Status
            big
            customLabel={i18n('Alarm fired')}
            statusType={StatusTypes.NO_SIGNAL}
        />
    ) : alarmSystem ? (
        <Status
            big
            customLabel={i18n('On')}
            statusType={StatusTypes.RIDING}
        />
    ) : (
        <Status
            big
            customLabel={i18n('Off')}
            statusType={StatusTypes.ENGINE_OFF}
        />
    );

    return [
        {
            key: i18n('Updated'),
            content: lastUpdate ? getLastUpdateString(lastUpdate) : EMPTY_DATA,
        },

        {
            key: i18n('Status'),
            content: (
                <CarStatus
                    big
                    carInfo={carInfo}
                />
            ),
        },

        ...(isCarAlarmSystemFlag()
            ? [
                  {
                      key: i18n('Alarm system'),
                      content: alertText,
                  },
              ]
            : []),
        ...(showCarScoring()
            ? [
                  {
                      key: i18n('Scoring'),
                      content: <Scoring value={aggressiveRank} />,
                  },
              ]
            : []),
        {
            key: i18n('Current speed'),
            content: speed,
        },

        {
            key: i18n('Fuel'),
            content: fuel_level !== null && typeof fuel_level === 'number' ? `${fuel_level.toFixed(1)} %` : EMPTY_DATA,
        },

        {
            key: i18n('Range'),
            content: fuel_distance ? `${fuel_distance} ${i18n('km')}` : EMPTY_DATA,
        },

        {
            key: i18n('Battery charge'),
            content: ext_voltage !== null ? `${ext_voltage.toFixed(1)} V` : EMPTY_DATA,
        },

        {
            key: i18n('Engine temp.'),
            content:
                engine_temperature !== null ? (
                    carInfo?.status &&
                    carInfo.status !== StatusTypes.ENGINE_ON &&
                    carInfo.status !== StatusTypes.RIDING &&
                    carTelematics?.engine_temperature_updated_at ? (
                        <div className={style.temp_engine_off}>
                            {`${engine_temperature} °C` +
                                ` (${getLastUpdateString(carTelematics.engine_temperature_updated_at)})`}
                        </div>
                    ) : (
                        `${engine_temperature} °C`
                    )
                ) : (
                    EMPTY_DATA
                ),
        },

        {
            key: i18n('GPS signal'),
            content: (
                <GPSValueIcon
                    GPSInview={GPSInview}
                    GPSUsed={GPSUsed}
                />
            ),
        },

        {
            key: i18n('GSM signal'),
            content:
                gsm_signal_level !== null ? (
                    <div>
                        <GSMSignalIcon GSMValue={gsm_signal_level} />
                    </div>
                ) : (
                    EMPTY_DATA
                ),
        },

        null,
        {
            key: i18n('Mileage'),
            content: mileage !== null ? `${mileage.toLocaleString()} ${i18n('km')}` : EMPTY_DATA,
        },
    ];
};
