import * as React from 'react';
import { useHistory } from 'react-router-dom';
import cn from 'classnames/bind';

import { MAP_DEFAULT_LAT, MAP_DEFAULT_LON, MAP_DEFAULT_ZOOM } from 'constants/constants';

import { BEACON_SEARCH_STATE } from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/BeaconSwitchMode/types';
import ShowOnMapButton from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/ShowOnMapButton';
import { StatusTypes } from 'components/types';
import { StackedButtons } from 'components/ui/Buttons/StackedButtons';

import { i18n } from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/SingleCarMap/index.i18n';

import BeaconEnabledLegendIcon from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/SingleCarMap/beacon_enabled_legend.component.svg';
import BeaconLegendIcon from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/SingleCarMap/beacon_legend.component.svg';
import GPSLegendIcon from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/SingleCarMap/gps_legend.component.svg';
import GPSNoSignalLegendIcon from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/SingleCarMap/gps_no_signal.component.svg';

import style from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/index.css';

const CAR_ZOOM = 14;

const cx = cn.bind(style);

export interface SingleCarMapProps {
    lon: number | null;
    lat: number | null;
    carInfo: any;
    beacon: any;
    isLoading: boolean;
    hideButtons?: boolean;
    className?: string;
    buttonsClassName?: string;
}

const CarsMap = React.lazy(() => import('components/Map/CarsMap'));
export const SingleCarMap = ({
    lon,
    lat,
    carInfo,
    beacon,
    isLoading,
    hideButtons,
    className,
    buttonsClassName,
}: SingleCarMapProps) => {
    const [shouldShowBeaconOnMap, showBeaconOnMap] = React.useState(false);
    let { status = StatusTypes.UNKNOWN } = carInfo ?? {};
    const history = useHistory();

    const hasCoords = Boolean(lon && lat);

    const onShowOnMap = () => {
        let urlSearchParams = new URLSearchParams();
        urlSearchParams.set('cars_view', 'map');
        urlSearchParams.set('zoom', CAR_ZOOM.toString());
        urlSearchParams.set('lon', lon ? lon.toString() : '');
        urlSearchParams.set('lat', lat ? lat.toString() : '');

        history.push(`/cars?${urlSearchParams}`);
    };

    return (
        <div
            className={cx(style.map_container, [className])}
            data-testid="map"
        >
            {hasCoords && !isLoading ? (
                <>
                    <React.Suspense fallback={<div />}>
                        <CarsMap
                            withoutTooltips
                            alwaysCarCenter
                            disableDrag={!shouldShowBeaconOnMap}
                            beacon={shouldShowBeaconOnMap ? beacon : undefined}
                            options={{
                                zoom: hasCoords ? CAR_ZOOM : MAP_DEFAULT_ZOOM,
                                center: hasCoords ? [lon!, lat!] : [MAP_DEFAULT_LON, MAP_DEFAULT_LAT],
                                hideBoundControls: true,
                                hideZoomControls: true,
                            }}
                            cars={carInfo ? [carInfo] : []}
                        />
                    </React.Suspense>
                    {!hideButtons && (
                        <div className={cx(style.map_button_container, [buttonsClassName])}>
                            <ShowOnMapButton
                                title={i18n('Show on map')}
                                onClick={onShowOnMap}
                            />

                            {beacon?.location?.longitude !== undefined && beacon?.location?.latitude !== undefined ? (
                                <StackedButtons
                                    buttons={[
                                        {
                                            icon:
                                                status === StatusTypes.NO_SIGNAL ? (
                                                    <GPSNoSignalLegendIcon />
                                                ) : (
                                                    <GPSLegendIcon />
                                                ),

                                            id: 'gps',
                                            disabled: true,
                                            active: true,
                                            tooltipContent: i18n('GPS location'),
                                        },

                                        {
                                            icon:
                                                beacon?.beacon_search_state == BEACON_SEARCH_STATE.ENABLED ? (
                                                    <BeaconEnabledLegendIcon />
                                                ) : (
                                                    <BeaconLegendIcon />
                                                ),

                                            onClick: () => {
                                                showBeaconOnMap((prev) => !prev);
                                            },
                                            id: 'beacon',
                                            tooltipContent: i18n('Beacon location'),
                                        },
                                    ]}
                                />
                            ) : null}
                        </div>
                    )}
                </>
            ) : null}
        </div>
    );
};
