import React, { useEffect, useState } from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import { MODAL_OBJECT_ID_CGI, MODAL_OBJECT_TYPE_CGI } from 'constants/constants';

import { MetrikaLabel } from 'shared/consts/MetrikaLabel';

import { BeaconSwitchMode } from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/BeaconSwitchMode';
import CarControls from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/CarControls';
import { CarLocation } from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/CarLocation';
import { CarTabInformation } from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/CarTabInformation';
import { getBeaconTab } from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/CarTabInformation/BeaconTab';
import { getCarTab } from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/CarTabInformation/CarTab';
import { getTelematicsTab } from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/CarTabInformation/TelematicsTab';
import { SingleCarMap } from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/SingleCarMap';
import { IBeacon } from 'components/Map/controls/BeaconPinLayout/types';
import { ModalObjectTypes } from 'components/types';
import ControlButton, { ButtonSize } from 'components/ui/Buttons/ControlButton';
import TabsHeader from 'components/ui/TabsHeader';

import { i18n } from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/index.i18n';

import Edit from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/edit.component.svg';

import style from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/index.css';

interface ICarMainWidget {
    carInfo: any;
    carTelematics: any;
    isLoading: boolean;
    beaconInfo: IBeacon | null;
}

enum CAR_WIDGET_MODES {
    TELEMATICS = 'telematics',
    CAR = 'car',
    BEACON = 'beacon',
}

const CAR_WIDGET_MODE = 'car_widget_mode';

export const CarMainWidget = ({ carInfo, carTelematics, isLoading, beaconInfo }: ICarMainWidget) => {
    const [tab, setTab] = useState<CAR_WIDGET_MODES | string>(CAR_WIDGET_MODES.TELEMATICS);

    let browserLocation = useLocation();
    let history = useHistory();

    const switchTab = (tab?: CAR_WIDGET_MODES | string | undefined) => {
        let searchParams = new URLSearchParams(browserLocation?.search);
        let selectedTab = tab || (searchParams.get(CAR_WIDGET_MODE) as CAR_WIDGET_MODES);
        if (tab) {
            searchParams.set(CAR_WIDGET_MODE, tab);
            history.push(`${browserLocation.pathname}?${searchParams}`);
        }
        selectedTab && setTab(selectedTab);
    };

    useEffect(() => {
        switchTab();
    }, [browserLocation]);

    let { lon = null, lat = null } = carInfo?.location ?? {};

    let getInfoArray = React.useCallback<
        () => Array<{
            key: string;
            content?: JSX.Element | string | null;
        } | null> | null
    >(() => {
        switch (tab) {
            case CAR_WIDGET_MODES.TELEMATICS:
                return getTelematicsTab({ carTelematics, carInfo });
            case CAR_WIDGET_MODES.CAR:
                return getCarTab({ carInfo });
            case CAR_WIDGET_MODES.BEACON:
                return getBeaconTab({ beaconInfo });
            default:
                return null;
        }
    }, [carInfo, carTelematics, tab]);

    const tabs = [
        { link: CAR_WIDGET_MODES.TELEMATICS, name: i18n('Telematics') },
        { link: CAR_WIDGET_MODES.CAR, name: i18n('Car') },
    ];

    if (beaconInfo) {
        tabs.splice(1, 0, { link: CAR_WIDGET_MODES.BEACON, name: i18n('Beacon') });
    }

    const onEditButtonClick = () => {
        let searchParams = new URLSearchParams(browserLocation.search);
        searchParams.set(MODAL_OBJECT_TYPE_CGI, ModalObjectTypes.UPSERT_CAR);
        searchParams.set(MODAL_OBJECT_ID_CGI, carInfo.id ?? '');
        history.push(`${browserLocation.pathname}?${searchParams}`);
    };

    return (
        <div className={style.car_main_widget_container}>
            <div className={`${style.car_main_widget} ${isLoading ? style.is_loading : ''}`}>
                <CarLocation
                    lon={lon}
                    lat={lat}
                    carInfo={carInfo}
                    isLoading={isLoading}
                />

                <SingleCarMap
                    beacon={beaconInfo}
                    lon={lon}
                    lat={lat}
                    carInfo={carInfo}
                    isLoading={isLoading}
                />

                <div className={style.car_tabs}>
                    <TabsHeader
                        activeTab={tab}
                        tabs={tabs}
                        onChange={switchTab}
                    />
                </div>
                <div className={style.car_info}>
                    {tab === CAR_WIDGET_MODES.BEACON && carInfo?.id && beaconInfo?.imei ? (
                        <BeaconSwitchMode
                            car_id={carInfo.id}
                            imei={beaconInfo.imei}
                            operation_mode={beaconInfo.operation_mode}
                            beacon_search_state={beaconInfo?.beacon_search_state}
                        />
                    ) : null}
                    {getInfoArray() ? (
                        <CarTabInformation
                            carInfo={carInfo}
                            isLoading={isLoading}
                            getInfoArray={getInfoArray}
                        />
                    ) : null}
                    {tab === CAR_WIDGET_MODES.CAR ? (
                        <div className={style.edit_button}>
                            <ControlButton
                                metrikaLabel={MetrikaLabel.EDIT_CAR}
                                size={ButtonSize.M}
                                tertiary
                                fullWidth
                                leftAlign
                                title={
                                    <>
                                        <Edit />
                                        {i18n('Edit')}
                                    </>
                                }
                                onClick={onEditButtonClick}
                            />
                        </div>
                    ) : null}
                </div>
                <CarControls
                    isLoading={isLoading}
                    carInfo={carInfo}
                />
            </div>
        </div>
    );
};
