import React from 'react';
import { useHistory, useParams } from 'react-router-dom';

import { ICarInfo } from 'utils/car/types';
import { isCarMaintenanceFlag } from 'utils/isCarMaintenanceFlag';

import { CarMaintenance } from 'features/CarMaintenance';

import { Path } from 'shared/consts/Path';
import { generateRouterPath } from 'shared/helpers/generateRouterPath/generateRouterPath';
import { ContentContainer } from 'shared/ui/ContentContainer/ContentContainer';
import { Widget } from 'shared/ui/Widget/Widget';

import CarCardInformers from 'components/Cars/CarCard/CarCardOverview/CarCardInformers';
import { CarMainWidget } from 'components/Cars/CarCard/CarCardOverview/CarMainWidget';
import CarSessionsTable from 'components/Cars/CarCard/CarSessionsTable';
import CarSignalsTable from 'components/Cars/CarCard/CarSignalsTable';
import { IBeacon } from 'components/Map/controls/BeaconPinLayout/types';

import { i18n } from 'components/Cars/CarCard/CarCardOverview/index.i18n';

import style from 'components/Cars/CarCard/CarCardOverview/index.css';

const MAX_TABLE_ROWS_LENGTH = 3;

interface ICarCardOverviewProps {
    carInfo: ICarInfo | null;
    carTelematics: any;
    isLoading: boolean;
    beaconInfo: IBeacon | null;
}

export const CarCardOverview = ({ carInfo, isLoading, carTelematics, beaconInfo }: ICarCardOverviewProps) => {
    const history = useHistory();
    const params = useParams();

    const onSignalsGetMore = React.useCallback(() => {
        history.push(generateRouterPath(Path.CAR_SIGNALS, params));
    }, [history, params]);

    const onRidesGetMore = React.useCallback(() => {
        history.push(generateRouterPath(Path.CAR_RIDES, params));
    }, [history, params]);

    return (
        <div className={style.car_card_overview}>
            <div className={style.main_widget_container}>
                <CarMainWidget
                    beaconInfo={beaconInfo}
                    isLoading={isLoading}
                    carInfo={carInfo}
                    carTelematics={carTelematics}
                />
            </div>

            <ContentContainer className={style.sessions_table_container}>
                {isCarMaintenanceFlag() && carInfo?.vin ? (
                    <div className={style.widget_item}>
                        <Widget>
                            <CarMaintenance vin={carInfo?.vin} />
                        </Widget>
                    </div>
                ) : null}

                <div className={style.widget_item}>
                    <Widget>
                        <div className={style.table_title}>{i18n('Signals')}</div>
                        <CarSignalsTable
                            numDoc={MAX_TABLE_ROWS_LENGTH}
                            drawColumns={['date_time', 'duration', 'event', 'session']}
                            getMore={{
                                canGetMore: true,
                                onGetMoreClick: onSignalsGetMore,
                                getMoreTitle: i18n('Show all'),
                            }}
                        />
                    </Widget>
                </div>

                <div className={style.widget_item}>
                    <Widget>
                        <div className={style.table_title}>{i18n('Rides')}</div>
                        <CarSessionsTable
                            numDoc={MAX_TABLE_ROWS_LENGTH}
                            getMore={{
                                canGetMore: true,
                                onGetMoreClick: onRidesGetMore,
                                getMoreTitle: i18n('Show all'),
                            }}
                        />
                    </Widget>
                </div>

                <CarCardInformers carInfo={carInfo} />
            </ContentContainer>
        </div>
    );
};
