import React, { MutableRefObject, useEffect, useRef, useState } from 'react';
import { useHistory, useLocation, useParams } from 'react-router-dom';

import {
    EMPTY_DATA,
    MODAL_OBJECT_ID_CGI,
    MODAL_OBJECT_TYPE_CGI,
    VERY_DANGEROUS_SESSION_BORDER,
} from 'constants/constants';

import getDuration from 'utils/sessions/getDuration';
import getFinish from 'utils/sessions/getFinish';
import getSessionId from 'utils/sessions/getSessionId';
import getStart from 'utils/sessions/getStart';
import { ISession } from 'utils/sessions/types';
import { showCarDriverInRides } from 'utils/showCarDriverInRides';
import { getUserFullName } from 'utils/user/getUserFullName';

import { FilterDateOptions } from 'features/FilterDate/ui/FilterDate/FilterDate';

import { getSessionDurationString } from 'components/Cars/CarCard/CarSessionsTable/getSessionDurationString';
import { getSessionMileageString } from 'components/Cars/CarCard/CarSessionsTable/getSessionMileageString';
import { SessionTime } from 'components/Cars/CarCard/CarSessionsTable/sessionTime';
import { CARS_REQUESTS, REQUESTS } from 'components/Cars/request';
import { ModalObjectTypes } from 'components/types';
import { Label, LabelIndicatorTypes } from 'components/ui/Label';
import Table from 'components/ui/Table';
import { ITableGetMore, ITableRowData } from 'components/ui/Table/types';

import { ABORT_ERROR_KEY, RequestHelper } from '../../../../../request-helper/src';

import { i18n } from 'components/Cars/CarCard/CarSessionsTable/index.i18n';

import style from 'components/Cars/CarCard/CarSessionsTable/index.css';

export type CarSessionsFiltersOptions = FilterDateOptions;

interface ICarSessionsTableProps {
    numDoc?: number;
    getMore?: ITableGetMore | null;
    filters?: CarSessionsFiltersOptions;
}

const SESSION_NUMDOC = 50;

// TODO: replace by RidesTable + RidesContentHeader
const CarSessionsTable = ({ numDoc, getMore, filters }: ICarSessionsTableProps) => {
    const request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: CARS_REQUESTS }));
    const [isLoadingSessions, setIsLoadingSessions] = useState<boolean>(true);
    const [hasMore, setHasMore] = useState<boolean>(false);
    const [isMoreLoadingSessions, setIsMoreLoadingSessions] = useState<boolean>(true);
    let { id } = useParams<{ id: string }>();
    let location = useLocation();
    let history = useHistory();

    const [sessions, setSessions] = useState<ISession[]>([]);
    const [sessionsError, setSessionsError] = useState<Error | null>(null);

    const CAR_SESSIONS_TABLE_HEADER = [
        { key: 'date_time', displayName: i18n('Date and time') },
        { key: 'duration', displayName: i18n('Duration') },
        { key: 'mileage', displayName: i18n('Mileage') },
        { key: 'violations', displayName: i18n('Violations') },
    ];

    if (showCarDriverInRides()) {
        CAR_SESSIONS_TABLE_HEADER.splice(1, 0, { key: 'driver', displayName: i18n('Driver') });
    }

    useEffect(() => {
        getSessions();

        return () => {
            request.current.abort();
        };
    }, []);

    useEffect(() => {
        request.current.abort();

        getSessions();
    }, [id, filters]);

    const getSessions = (more?: boolean) => {
        if (more) {
            setIsMoreLoadingSessions(true);
        } else {
            setIsLoadingSessions(true);
        }

        let lastSessionStart: number | null = null;
        if (more) {
            let lastSession = sessions.length ? sessions[sessions.length - 1] : null;
            if (lastSession) {
                lastSessionStart = getStart(lastSession)?.timestamp ?? null;
            }
        }

        return request.current
            .exec(REQUESTS.GET_CAR_SESSIONS, {
                queryParams: {
                    car_id: id,
                    numdoc: numDoc ?? SESSION_NUMDOC,
                    since: more ? null : filters?.since ?? null,
                    until: more ? lastSessionStart ?? null : filters?.until ?? null,
                },
            })
            .then((response) => {
                if (response.meta !== ABORT_ERROR_KEY) {
                    let { sessions: sessionResponse = [] } = response;
                    setSessions(more ? [...sessions, ...sessionResponse] : sessionResponse);
                    setIsLoadingSessions(false);
                    setIsMoreLoadingSessions(false);
                    setHasMore(response.has_more);
                }
            })
            .catch((error) => {
                setSessionsError(error);
                setIsLoadingSessions(false);
                setIsMoreLoadingSessions(false);
            });
    };

    const onRowClick = (sessionId: string) => {
        let searchParams = new URLSearchParams(location.search);
        searchParams.set(MODAL_OBJECT_TYPE_CGI, ModalObjectTypes.SESSION);
        searchParams.set(MODAL_OBJECT_ID_CGI, sessionId);
        history.push(`${location.pathname}?${searchParams}`);
    };

    const buildTableRows = (): ITableRowData[] => {
        return sessions.map((session, index) => {
            let session_id = getSessionId(session);

            return {
                data: {
                    date_time: (
                        <SessionTime
                            start={getStart(session)?.timestamp}
                            finish={getFinish(session)?.timestamp}
                        />
                    ),

                    driver: session.user_details ? (
                        <Label
                            background
                            title={getUserFullName(session.user_details)}
                        />
                    ) : (
                        EMPTY_DATA
                    ),

                    duration: getSessionDurationString(getDuration(session)),
                    mileage: getSessionMileageString(session),
                    violations: (
                        <div className={style.label_container}>
                            {session?.is_speeding && (
                                <Label
                                    background
                                    indicator={{ type: LabelIndicatorTypes.WARNING }}
                                    title={i18n('Speeding')}
                                />
                            )}

                            {session?.is_aggressive ? (
                                session?.aggressive_score ? (
                                    session?.aggressive_score < VERY_DANGEROUS_SESSION_BORDER ? (
                                        <Label
                                            background
                                            indicator={{ type: LabelIndicatorTypes.WARNING }}
                                            title={i18n('Dangerous ride')}
                                        />
                                    ) : (
                                        <Label
                                            background
                                            indicator={{ type: LabelIndicatorTypes.ERROR }}
                                            title={i18n('Very dangerous ride')}
                                        />
                                    )
                                ) : (
                                    <Label
                                        background
                                        indicator={{ type: LabelIndicatorTypes.ERROR }}
                                        title={i18n('Careless driving')}
                                    />
                                )
                            ) : null}
                        </div>
                    ),
                },

                meta: {
                    key: session_id ?? index.toString(),
                    className: style.car_session_row,
                    onClick: onRowClick.bind(null, session_id),
                },
            };
        });
    };

    return (
        <Table
            header={CAR_SESSIONS_TABLE_HEADER}
            tableData={buildTableRows()}
            isLoading={isLoadingSessions}
            error={sessionsError}
            getMore={
                getMore ?? hasMore
                    ? {
                          canGetMore: getMore?.canGetMore ? getMore.canGetMore : hasMore,
                          onGetMoreClick: getMore?.onGetMoreClick
                              ? getMore?.onGetMoreClick
                              : hasMore
                              ? getSessions.bind(null, true)
                              : null,
                          getMoreTitle: getMore?.getMoreTitle ? getMore.getMoreTitle : undefined,
                          isMoreLoading: isMoreLoadingSessions,
                      }
                    : null
            }
            reloadFunction={getSessions}
        />
    );
};

export default CarSessionsTable;
