import React, { MutableRefObject, useEffect, useRef, useState } from 'react';
import { useParams } from 'react-router-dom';

import { SIGNALS_PAGE_SIZE, VERY_DANGEROUS_SESSION_BORDER } from 'constants/constants';

import { SCORING_TRACE_TAG } from 'components/Cars/constants';
import { REQUESTS, SIGNALS_REQUESTS } from 'components/Signals/request';
import SignalsTable from 'components/Signals/SignalsTable';
import { FormattedSignal, ISignalsResponse, TableHeadersKeys } from 'components/Signals/types';
import { ITableGetMore } from 'components/ui/Table/types';

import { ABORT_ERROR_KEY, RequestHelper } from '../../../../../request-helper/src';

interface ICarSessionsTableProps {
    numDoc?: number;
    getMore?: ITableGetMore | null;
    drawColumns?: TableHeadersKeys[];
}

const CarSignalsTable = ({ numDoc, getMore, drawColumns }: ICarSessionsTableProps) => {
    const [isSignalsLoading, setIsSignalsLoading] = useState<boolean>(true);
    const [signalsError, setSignalsError] = useState<Error | null>(null);
    const [formattedSignals, setFormattedSignals] = useState<FormattedSignal[]>([]);
    const [canGetMore, setCanGetMore] = useState<boolean>(false);
    const [isMoreLoading, setIsMoreLoading] = useState<boolean>(true);
    const [nextCarsCursor, setNextCarsCursor] = useState<number | null>(null);
    const [nextSessionsCursor, setNextSessionsCursor] = useState<number | null>(null);
    let { id } = useParams<{ id: string }>();
    const request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: SIGNALS_REQUESTS }));

    useEffect(() => {
        getSignals();

        return () => {
            request.current.abort();
        };
    }, []);

    useEffect(() => {
        request.current.abort();

        getSignals();
    }, [id]);

    const getSignals = (isMore?: boolean) => {
        if (isMore) {
            setIsMoreLoading(true);
        } else {
            setIsSignalsLoading(true);
        }

        setSignalsError(null);
        request.current.abort();
        request.current
            .exec(REQUESTS.GET_SIGNALS, {
                queryParams: {
                    cars_ids: id,
                    page_size: numDoc ?? SIGNALS_PAGE_SIZE,
                    cars_cursor: isMore ? nextCarsCursor : null,
                    sessions_cursor: isMore ? nextSessionsCursor : null,
                },
            })
            .then((signalsResponse: ISignalsResponse) => {
                let { signals, signals_descriptions, can_get_more_pages, next_cars_cursor, next_sessions_cursor } =
                    signalsResponse;

                let formattedSignalsResponse = signals
                    ? signals.map((signal) => {
                          let { name, details } = signal;
                          let description = { ...(signals_descriptions?.[name] || {}) };
                          let formattedSignal: FormattedSignal = Object.assign({}, { description }, signal);

                          if (name === SCORING_TRACE_TAG && formattedSignal.description) {
                              let sessionScore = details.score ?? 0;
                              formattedSignal.description.display_name =
                                  sessionScore < VERY_DANGEROUS_SESSION_BORDER
                                      ? 'Опасная поездка'
                                      : 'Очень опасная поездка';
                              formattedSignal.description.priority =
                                  sessionScore < VERY_DANGEROUS_SESSION_BORDER ? 'warning' : 'critical';
                          }

                          return formattedSignal;
                      })
                    : [];

                setIsMoreLoading(signalsResponse.meta === ABORT_ERROR_KEY);
                setIsSignalsLoading(signalsResponse.meta === ABORT_ERROR_KEY);
                setCanGetMore(can_get_more_pages ?? false);
                setNextCarsCursor(next_cars_cursor ?? null);
                setNextSessionsCursor(next_sessions_cursor ?? null);

                setFormattedSignals(
                    isMore ? [...formattedSignals, ...formattedSignalsResponse] : formattedSignalsResponse,
                );
            })
            .catch((error) => {
                setIsMoreLoading(false);
                setIsSignalsLoading(false);
                setSignalsError(error);
            });
    };

    return (
        <SignalsTable
            isLoading={isSignalsLoading}
            error={signalsError}
            signals={formattedSignals}
            drawColumns={drawColumns}
            getSignals={getSignals}
            getMore={
                getMore ?? {
                    canGetMore,
                    isMoreLoading,
                    onGetMoreClick: getSignals.bind(null, true),
                }
            }
        />
    );
};

export default CarSignalsTable;
