import React, { MutableRefObject, useEffect, useRef, useState } from 'react';
import { useParams } from 'react-router-dom';

import { EMPTY_DATA } from 'constants/constants';

import shortDateTime from 'utils/date/shortDateTime';

import { Widget } from 'shared/ui/Widget/Widget';

import { REQUESTS, TELEMATICS_REQUESTS } from 'components/Cars/CarCard/CarTelematicsTable/request';
import Table from 'components/ui/Table';
import { ITableRowData } from 'components/ui/Table/types';

import { ABORT_ERROR_KEY, RequestHelper } from '../../../../../request-helper/src';

import { i18n } from 'components/Cars/CarCard/CarTelematicsTable/index.i18n';

import style from 'components/Cars/CarCard/CarTelematicsTable/index.css';

const UPDATE_TIME = 1500;

interface ITelematic {
    id: number;
    subid?: number;
    name: string;
    since: number;
    updated: number;
    value: number;
}

const CarTelematicsTable = () => {
    const [isLoadingTelematics, setIsLoadingTelematics] = useState<boolean>(true);
    const [telematics, setTelematics] = useState<ITelematic[]>([]);
    const [telematicsError, setTelematicsError] = useState<Error | null>(null);
    let { id } = useParams<{ id: string }>();

    const request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: TELEMATICS_REQUESTS }));
    const updateTimer: MutableRefObject<any> = useRef(null);

    const CAR_SESSIONS_TABLE_HEADER = [
        { key: 'id', displayName: 'ID' },
        { key: 'sensor', displayName: i18n('Sensor') },
        { key: 'value', displayName: i18n('Value') },
        { key: 'start', displayName: i18n('Start') },
        { key: 'updated', displayName: i18n('Updated') },
    ];

    useEffect(() => {
        setIsLoadingTelematics(true);
        getTelematics(true);

        return () => {
            request.current.abort();
            clearTimeout(updateTimer.current);
        };
    }, []);

    const getTelematics = (initialRun?: boolean) => {
        setTelematicsError(null);
        request.current
            .exec(REQUESTS.GET_TELEMATICS, {
                queryParams: {
                    car_id: id,
                },
            })
            .finally(() => {
                setIsLoadingTelematics(false);
            })
            .then((response: { sensors: ITelematic[] } & { meta: string }) => {
                if (response.meta !== ABORT_ERROR_KEY) {
                    clearTimeout(updateTimer.current);
                    updateTimer.current = setTimeout(() => {
                        getTelematics();
                    }, UPDATE_TIME);

                    let { sensors = [] } = response;
                    setTelematics(sensors);
                }
            })
            .catch((error) => {
                clearTimeout(updateTimer.current);
                updateTimer.current = setTimeout(() => {
                    getTelematics();
                }, UPDATE_TIME);

                if (initialRun) {
                    setTelematicsError(error);
                }
            });
    };

    const getDateString = (timestamp: number | null) => {
        return timestamp ? shortDateTime(timestamp) : EMPTY_DATA;
    };

    const buildTableRows = (): ITableRowData[] => {
        return telematics.map((telematic, index) => {
            let { id, subid, name, value, since, updated } = telematic ?? {};

            return {
                data: {
                    id: `${id}${subid ? `/${subid}` : ''}`,
                    sensor: name,
                    value: (
                        <div
                            className={style.telematics_value_cell}
                            title={value?.toString() ?? ''}
                        >
                            {value}
                        </div>
                    ),

                    start: getDateString(since),
                    updated: getDateString(updated),
                },

                meta: {
                    key: `${id}${subid ? `/${subid}` : ''}` ?? index,
                },
            };
        });
    };

    return (
        <Widget contentContainer>
            <Table
                header={CAR_SESSIONS_TABLE_HEADER}
                tableData={buildTableRows()}
                isLoading={isLoadingTelematics}
                error={telematicsError}
                reloadFunction={getTelematics.bind(null, true)}
            />
        </Widget>
    );
};

export default CarTelematicsTable;
