import React, { ClassicElement, MutableRefObject, useContext, useEffect, useRef, useState } from 'react';
import { useLocation, useParams, useRouteMatch } from 'react-router-dom';

import { ONE_MINUTE } from 'constants/constants';

import FlagsContext from 'contexts/FlagsContext';

import { ICarInfo } from 'utils/car/types';

import { CAR_HEADER_TAB_ROUTES } from 'pages/Car/Redirect/ui/CarRedirectPage/CarRedirectPage';

import { CarHeader } from 'widgets/CarHeader/ui/CarHeader/CarHeader';
import { CarSessionsFilters } from 'widgets/CarSessionsFilters';
import { HEADER_HEIGHT } from 'widgets/Header';

import { useCar } from 'entities/Car/api/useCar/useCar';
import { CarCardTab } from 'entities/Car/consts/CarCardTab';
import { INITIAL_CAR_SESSIONS_FILTERS } from 'entities/Car/consts/filters';
import { CarNotFound } from 'entities/Car/ui/CarNotFound/CarNotFound';

import { ContentContainer } from 'shared/ui/ContentContainer/ContentContainer';
import { ErrorBoundary } from 'shared/ui/ErrorBoundary/ErrorBoundary';
import { SectionLayout } from 'shared/ui/SectionLayout/SectionLayout';
import { Widget } from 'shared/ui/Widget/Widget';

import { CarCardOverview } from 'components/Cars/CarCard/CarCardOverview';
import CarSessionsTable, { CarSessionsFiltersOptions } from 'components/Cars/CarCard/CarSessionsTable';
import CarSignalsTable from 'components/Cars/CarCard/CarSignalsTable';
import CarTelematicsTable from 'components/Cars/CarCard/CarTelematicsTable';
import { CARS_REQUESTS, REQUESTS } from 'components/Cars/request';
import { IBeacon } from 'components/Map/controls/BeaconPinLayout/types';

import { ABORT_ERROR_KEY, RequestHelper } from '../../../../request-helper/src';

export interface IFlagKey {
    key: string;
    value: boolean;
}

const UPDATE_CAR_TIMER = 1500;
const CAR_SENSORS = ['speed', 'mileage', 'fuel_level', 'fuel_distance', 'ext_voltage', 'engine_on', 'gsm_signal_level'];

const CarCard = () => {
    let { id } = useParams<{ id: string }>();

    const carResource = useCar({
        carId: id,
        sensors: CAR_SENSORS,
    });

    const [car, setCar] = useState<ICarInfo | null>(null);
    const [carTelematics, setCarTelematics] = useState<any>(null);
    const [beaconInfo, setBeaconInfo] = useState<IBeacon | null>(null);
    const [isInitialLoadingCar, setIsInitialLoadingCar] = useState<boolean>(true);
    const [currentView, setCurrentView] = useState<ClassicElement<any> | null>(null);
    const [currentFilters, setCurrentFilters] = useState<ClassicElement<any> | null>(null);
    const [updateTime, setUpdateTime] = useState<number>(UPDATE_CAR_TIMER);

    const [sessionsFilters, setSessionsFilters] = useState<CarSessionsFiltersOptions>(INITIAL_CAR_SESSIONS_FILTERS);

    const [flagKeys, setFlagKeys] = useState<IFlagKey[]>([]);
    let flags = useContext(FlagsContext);
    let location = useLocation();
    const request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: CARS_REQUESTS }));
    const timer: MutableRefObject<any> = useRef(null);
    const telematicsTimer: MutableRefObject<any> = useRef(null);
    const beaconTimer: MutableRefObject<any> = useRef(null);

    useEffect(() => {
        getCar(true);
        getCarTelematics();
        getCarBeacon();

        return () => {
            clearTimeout(timer.current);
            clearTimeout(telematicsTimer.current);
            clearTimeout(beaconTimer.current);
            request.current.abort();
        };
    }, []);

    useEffect(() => {
        clearTimeout(timer.current);
        clearTimeout(telematicsTimer.current);
        clearTimeout(beaconTimer.current);
        request.current.abort();

        getCar(true);
        getCarTelematics();
        getCarBeacon();
    }, [id, updateTime]);

    const onSessionsFiltersChange = (filters: CarSessionsFiltersOptions) => {
        setSessionsFilters(filters);
    };

    // @TODO убрать когда разьедимся на 4 страницы
    const selectedTab =
        Object.keys(CAR_HEADER_TAB_ROUTES).find((key) => {
            return useRouteMatch(CAR_HEADER_TAB_ROUTES[key]);
        }) || CarCardTab.OVERVIEW;

    useEffect(() => {
        let currentView: ClassicElement<any> | null = null;
        let currentFilters: ClassicElement<any> | null = null;

        switch (selectedTab) {
            case CarCardTab.OVERVIEW:
                currentView = (
                    <CarCardOverview
                        beaconInfo={beaconInfo}
                        carInfo={car}
                        carTelematics={carTelematics}
                        isLoading={isInitialLoadingCar}
                    />
                );
                break;
            case CarCardTab.SESSIONS:
                if (flagKeys.find((flagKey) => flagKey.key === 'sessions')?.value) {
                    currentView = (
                        <ContentContainer
                            bodyScroll
                            withSidebar
                        >
                            <Widget contentContainer>
                                <CarSessionsTable filters={sessionsFilters} />
                            </Widget>
                        </ContentContainer>
                    );

                    currentFilters = (
                        <CarSessionsFilters
                            offsetTop={HEADER_HEIGHT}
                            onFiltersChange={onSessionsFiltersChange}
                        />
                    );
                }
                break;
            case CarCardTab.TELEMATICS:
                if (flagKeys.find((flagKey) => flagKey.key === 'telematics')?.value) {
                    currentView = (
                        <ContentContainer bodyScroll>
                            <CarTelematicsTable />
                        </ContentContainer>
                    );
                }
                break;
            case CarCardTab.SIGNALS:
                currentView = (
                    <ContentContainer bodyScroll>
                        <Widget contentContainer>
                            <CarSignalsTable drawColumns={['date_time', 'duration', 'event', 'session']} />
                        </Widget>
                    </ContentContainer>
                );
                break;
        }

        setCurrentView(currentView);
        setCurrentFilters(currentFilters);
        setUpdateTime(selectedTab === CarCardTab.OVERVIEW ? UPDATE_CAR_TIMER : ONE_MINUTE);
    }, [location, selectedTab, isInitialLoadingCar, car, carTelematics, sessionsFilters]);

    const getCar = (initial?: boolean) => {
        if (initial) {
            setIsInitialLoadingCar(true);
        }

        request.current
            .exec(REQUESTS.GET_CARS, {
                queryParams: {
                    car_id: id,
                    sensors: CAR_SENSORS.join(','),
                    traits: [
                        'ReportShowAllModels',
                        'ReportLocationDetails',
                        'ReportAggressiveScore',
                        'ReportSensorTimestamp',
                        'ReportTagDetails',
                        'ReportVIN',
                        'ReportIMEI',
                    ].join(','),
                },
            })
            .finally(() => {
                setIsInitialLoadingCar(false);

                clearTimeout(timer.current);
                timer.current = setTimeout(() => {
                    getCar();
                }, updateTime);
            })
            .then((carInfo) => {
                if (carInfo.meta !== ABORT_ERROR_KEY) {
                    let { cars, models } = carInfo;
                    let car = cars?.[0] ?? {};

                    car.model = models?.[car?.model_id]?.name || car?.model_id;
                    car.number = car.number?.toUpperCase();
                    car.taxi_park = car?.taxi_company?.name;

                    if (!car?.telematics) {
                        car.telematics = {};
                    }

                    setCar(car);
                }
            });
    };

    const getCarTelematics = () => {
        request.current
            .exec(REQUESTS.GET_CAR_TELEMATICS, {
                queryParams: {
                    car_id: id,
                },
            })
            .finally(() => {
                clearTimeout(telematicsTimer.current);
                telematicsTimer.current = setTimeout(() => {
                    getCarTelematics();
                }, updateTime);
            })
            .then((telematicsInfo) => {
                let { sensors = [] } = telematicsInfo;

                let carTelematics = sensors.reduce((result, sensor) => {
                    let { name, value, updated } = sensor;
                    result[name] = value ?? null;
                    result[`${name}_updated_at`] = updated ?? null;

                    return result;
                }, {});

                setCarTelematics(carTelematics);
            });
    };

    const getCarBeacon = () => {
        request.current
            .exec(REQUESTS.GET_CAR_BEACON, {
                queryParams: {
                    car_id: id,
                },
            })
            .finally(() => {
                clearTimeout(beaconTimer.current);
                beaconTimer.current = setTimeout(() => {
                    getCarBeacon();
                }, updateTime);
            })
            .then((beaconInfo) => {
                let { beacons = [] } = beaconInfo;
                setBeaconInfo(beacons[0]);
            });
    };

    useEffect(() => {
        let showSessions = flags?.showSessions;
        let showTelematics = flags?.showTelematics;

        let flagKeys: IFlagKey[] = [
            { key: 'sessions', value: showSessions },
            { key: 'telematics', value: showTelematics },
        ];
        setFlagKeys(flagKeys);
    }, [flags]);

    const isBodyScroll = selectedTab !== CarCardTab.OVERVIEW;

    return (
        <ErrorBoundary fallback={CarNotFound}>
            <SectionLayout
                header={<CarHeader resource={carResource} />}
                filters={currentFilters}
                bodyScroll={isBodyScroll}
            >
                {currentView}
            </SectionLayout>
        </ErrorBoundary>
    );
};

export default CarCard;
