import React, { useEffect, useState } from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import { MODAL_OBJECT_ID_CGI, MODAL_OBJECT_TYPE_CGI } from 'constants/constants';

import { Shimmer } from 'shared/ui/Shimmer/Shimmer';

import { IGroup } from 'components/Cars/CarsGroups/types';
import { ModalObjectTypes } from 'components/types';
import Card from 'components/ui/Card';
import ErrorReloadLabel from 'components/ui/ErrorLabel/ErrorReloadLabel';
import Main2 from 'components/ui/Text/Main2';

import { i18n } from 'components/Cars/CarsGroups/CarsGroupsContent/index.i18n';

import style from 'components/Cars/CarsGroups/CarsGroupsContent/index.css';

interface ICarsGroupsContentProps {
    groups: IGroup[];
    isLoading: boolean;
    error: Error | null;
    getGroups: () => void;
}

const CarsGroupsContent = ({ groups, getGroups, isLoading, error }: ICarsGroupsContentProps) => {
    const [separatedGroups, setSeparatedGroups] = useState<{ user: IGroup[]; company: IGroup[] }>({
        user: [],
        company: [],
    });

    let location = useLocation();
    let history = useHistory();

    useEffect(() => {
        let separatedGroups = (groups ?? []).reduce(
            (
                result: {
                    user: IGroup[];
                    company: IGroup[];
                },

                group,
            ) => {
                if (group.visibility === 'company') {
                    result.company.push(group);
                } else {
                    result.user.push(group);
                }

                return result;
            },
            {
                user: [],
                company: [],
            },
        );

        setSeparatedGroups(separatedGroups);
    }, [groups]);

    const onCardClick = (id: string) => {
        let searchParams = new URLSearchParams(location.search);
        searchParams.set(MODAL_OBJECT_TYPE_CGI, ModalObjectTypes.CAR_GROUP);
        searchParams.set(MODAL_OBJECT_ID_CGI, id);
        history.push(`${location.pathname}?${searchParams}`);
    };

    const buildCardsGroup = (visibility: 'user' | 'company') => {
        let groups = separatedGroups[visibility] ?? [];
        if (!groups.length) {
            return null;
        }

        let title = visibility === 'user' ? i18n('For me') : i18n('For company');

        return (
            <div className={style.cards_group}>
                <Main2 className={style.cards_group_title}>{title}</Main2>
                <div className={style.reports_container}>
                    {groups.map((group) => {
                        let { id, display_name, description, type } = group;
                        let typeDisplayName: string | null = null;
                        if (type === 'car') {
                            typeDisplayName = i18n('Cars');
                        }

                        let labels = typeDisplayName ? [typeDisplayName] : [];

                        return (
                            <Card
                                key={id}
                                name={display_name}
                                labels={labels}
                                description={description}
                                onClick={onCardClick.bind(null, id)}
                            />
                        );
                    })}
                </div>
            </div>
        );
    };

    return error ? (
        <div className={style.no_data_message}>
            <ErrorReloadLabel reloadFunction={getGroups} />
        </div>
    ) : isLoading ? (
        <div className={style.cards_group}>
            <Shimmer className={style.title_shimmer} />
            <Shimmer className={style.shimmer} />
        </div>
    ) : groups.length ? (
        <div className={style.reports_groups_container}>
            {buildCardsGroup('user')}
            {buildCardsGroup('company')}
        </div>
    ) : (
        <div className={style.no_data_message}>
            <span className={style.title}>{i18n('Create your first group')}</span>
            <span className={style.advice}>{i18n('Filter the cars and save as a group. It will appear here.')}</span>
        </div>
    );
};

export default CarsGroupsContent;
