import React, { MutableRefObject, useEffect, useRef, useState } from 'react';

import { showOrganizationGroupInfo } from 'utils/showOrganizationGroupInfo';

import { CARS_GROUPS_REQUESTS, REQUESTS } from 'components/Cars/CarsGroups/request';
import { IGroup } from 'components/Cars/CarsGroups/types';
import ControlButton from 'components/ui/Buttons/ControlButton';
import ErrorLabel from 'components/ui/ErrorLabel';
import { Input } from 'components/ui/Input';
import { Modal } from 'components/ui/Modal';
import { RadioButton } from 'components/ui/RadioButton';
import Main2 from 'components/ui/Text/Main2';

import { RequestHelper } from '../../../../../request-helper/src';

import { i18n } from 'components/Cars/CarsGroups/GroupModal/index.i18n';

import style from 'components/Cars/CarsGroups/GroupModal/index.css';

export interface IGroupModalFormField {
    display_name: string;
    description: string;
    visibility?: 'user' | 'company';
}

export interface IGroupModalErrorField {
    display_name: string;
    description: string;
    visibility?: string;
}

interface IReportModalProps {
    onClose: (success?: boolean) => void;
    type?: string;
    conditions?: string;
}

const GroupModal = ({ onClose, type, conditions }: IReportModalProps) => {
    const [isLoading, setIsLoading] = useState<boolean>(false);
    const [error, setError] = useState<Error | null>(null);
    const [errorTitle, setErrorTitle] = useState<string | null>(null);
    const [formValue, setFormValue] = useState<IGroupModalFormField>({
        display_name: '',
        description: '',
        visibility: 'user',
    });

    const [formErrors, setFormErrors] = useState<IGroupModalErrorField>({
        display_name: '',
        description: '',
        visibility: '',
    });

    const request: MutableRefObject<RequestHelper> = useRef(
        new RequestHelper({ requestConfigs: CARS_GROUPS_REQUESTS }),
    );

    useEffect(() => {
        if (!conditions) {
            setError(new Error());
            setErrorTitle(i18n("Can't create group with empty filters"));
        }
    }, []);

    const onFormChange = (type: string, value: string) => {
        let newFormValue = Object.assign({}, formValue);
        let newFormErrors = Object.assign({}, formErrors);
        newFormValue[type] = value;
        newFormErrors[type] = null;

        setFormValue(newFormValue);
        setFormErrors(newFormErrors);

        if (!conditions) {
            setError(new Error());
            setErrorTitle(i18n("Can't create group with empty filters"));
        } else {
            setError(null);
        }
    };

    const getGroups = () => {
        return request.current
            .exec(REQUESTS.GET_GROUPS)
            .then((response: { objects: { company_named_filters: IGroup[]; self_named_filters: IGroup[] } }) => {
                return [
                    ...(response.objects.company_named_filters ?? []).map((group) => {
                        group.visibility = 'company';

                        return group;
                    }),
                    ...(response.objects.self_named_filters ?? []).map((group) => {
                        group.visibility = 'user';

                        return group;
                    }),
                ];
            });
    };

    const onAddClick = () => {
        setIsLoading(true);
        setError(null);
        setErrorTitle(null);

        getGroups()
            .then((groups: IGroup[]) => {
                let isDisplayNameExist = false;
                let isConditionExist = false;

                groups.forEach((group) => {
                    if (group.display_name === formValue.display_name && group.visibility === formValue.visibility) {
                        isDisplayNameExist = true;
                    }
                    if (group.filter?.conditions === conditions && group.visibility === formValue.visibility) {
                        isConditionExist = true;
                    }
                });

                if (isDisplayNameExist) {
                    let newFormErrors = Object.assign({}, formErrors);
                    newFormErrors.display_name = i18n('A group with the same name already exists');

                    setFormErrors(newFormErrors);
                    setIsLoading(false);
                } else if (isConditionExist) {
                    setError(new Error());
                    setErrorTitle(i18n('A group with such filters already exists'));
                    setIsLoading(false);
                } else if (!conditions) {
                    setError(new Error());
                    setErrorTitle(i18n("Can't create group with empty filters"));
                    setIsLoading(false);
                } else {
                    addGroup(formValue);
                }
            })
            .catch((error) => {
                setIsLoading(false);
                setError(error);
            });
    };

    const addGroup = (group: IGroup) => {
        let groupObject: IGroup = {
            display_name: group.display_name,
            description: group.description,
            filter: {
                conditions: conditions ?? '',
            },

            priority: 0,
        };

        if (type) {
            groupObject.type = type;
        }

        let data = {
            object: groupObject,
        };

        request.current
            .exec(REQUESTS.ADD_GROUP, {
                body: data,
                queryParams: { to_organization: formValue.visibility === 'company' },
            })
            .finally(() => {
                setIsLoading(false);
            })
            .then(() => {
                onClose(true);
            })
            .catch(setError);
    };

    return (
        <Modal
            onClose={onClose}
            title={i18n('Create new group')}
            contentComponent={
                <div>
                    <div className={style.inputs_container}>
                        <Input
                            value={formValue.display_name}
                            label={i18n('Group name')}
                            error={formErrors.display_name}
                            onChange={onFormChange.bind(null, 'display_name')}
                        />

                        {showOrganizationGroupInfo() ? (
                            <div className={style.switcher_container}>
                                <Main2 className={style.title}>{i18n('Access')}</Main2>
                                <RadioButton
                                    secondary
                                    fullWidth
                                    values={[
                                        {
                                            value: 'user',
                                            placeholder: i18n('Only for me'),
                                        },

                                        {
                                            value: 'company',
                                            placeholder: i18n('For company'),
                                        },
                                    ]}
                                    onChange={onFormChange.bind(null, 'visibility')}
                                />
                            </div>
                        ) : null}
                        <Input
                            value={formValue.description}
                            label={i18n('Description')}
                            error={formErrors.description}
                            onChange={onFormChange.bind(null, 'description')}
                        />
                    </div>
                    <div className={style.buttons_container}>
                        {error ? (
                            <ErrorLabel
                                simple
                                title={errorTitle ?? ''}
                                className={style.error_container}
                            />
                        ) : null}
                        <ControlButton
                            fullWidth
                            disabled={!formValue.display_name || isLoading}
                            title={i18n('Create')}
                            onClick={onAddClick}
                        />
                    </div>
                </div>
            }
        />
    );
};

export default GroupModal;
