import React, { MutableRefObject, useEffect, useRef, useState } from 'react';

import { isValidJSONString } from 'utils/isValidJSONString';

import CustomizeTableItem from 'components/Cars/CarsSidebar/CustomizeTableButton/CustomizeTableModal/CustomizeTableItem';
import { REQUESTS, SETTINGS_REQUESTS } from 'components/Reports/request';
import ControlButton from 'components/ui/Buttons/ControlButton';
import ErrorLabel from 'components/ui/ErrorLabel';
import ErrorReloadLabel from 'components/ui/ErrorLabel/ErrorReloadLabel';
import { Modal } from 'components/ui/Modal';
import { Separator } from 'components/ui/SectionLayout/SidebarFilterItem';
import Select from 'components/ui/Select';
import { IHeaderInfo } from 'components/ui/Table/types';

import { RequestHelper } from '../../../../../../request-helper/src';

import { i18n } from 'components/Cars/CarsSidebar/CustomizeTableButton/CustomizeTableModal/index.i18n';

import style from 'components/Cars/CarsSidebar/CustomizeTableButton/CustomizeTableModal/index.css';

export const TABLE_VIEW_SETTINGS_KEY_PREFIX = 'table_view_';

interface ICustomizeTableModalProps {
    onCloseModal: (columns?: string[]) => void;
    defaultColumns: string[];
    tableType: string;
    headers: IHeaderInfo[];
    requiredColumns: string[];
    filterColumns?: (arr: string[]) => string[];
}

const CustomizeTableModal = ({
    onCloseModal,
    defaultColumns,
    tableType,
    headers,
    requiredColumns,
    filterColumns = (arr) => arr,
}: ICustomizeTableModalProps) => {
    const [isLoading, setIsLoading] = useState<boolean>(false);
    const [loadingError, setLoadingError] = useState<Error | null>(null);
    const [tableSettings, setTableSettings] = useState<{ columns: string[] } | null>(null);
    const [isSaving, setSaving] = useState<boolean>(false);
    const [savingError, setSavingError] = useState<Error | null>(null);

    const request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: SETTINGS_REQUESTS }));

    useEffect(() => {
        getTableSettings();
    }, []);

    const onAddColumn = (newColumn: string) => {
        let newTableSettings = { columns: [...(tableSettings?.columns || []), newColumn] };
        setTableSettings(newTableSettings);
    };

    const onRemoveColumn = (removeColumn: string) => {
        let newTableSettings = {
            columns: [...(tableSettings?.columns.filter((column) => column !== removeColumn) || [])],
        };

        setTableSettings(newTableSettings);
    };

    const onResetSettings = () => {
        setTableSettings({
            columns: defaultColumns,
        });
    };

    const getTableSettings = () => {
        setIsLoading(true);
        setLoadingError(null);
        request.current
            .exec(REQUESTS.GET_SETTINGS)
            .finally(() => {
                setIsLoading(false);
            })
            .then((settingsArray) => {
                let tableViewSettings =
                    settingsArray?.settings?.find(
                        (settingsItem) => settingsItem.id === `${TABLE_VIEW_SETTINGS_KEY_PREFIX}${tableType}`,
                    ) ?? [];
                let tableSettings = isValidJSONString(tableViewSettings.value)
                    ? JSON.parse(tableViewSettings.value)?.[0]
                    : null;

                setTableSettings({
                    columns: tableSettings?.columns ?? defaultColumns,
                });
            })
            .catch(setLoadingError);
    };

    const onSaveClick = () => {
        setSaving(true);
        setSavingError(null);
        request.current
            .exec(REQUESTS.SET_SETTINGS, {
                body: {
                    id: `${TABLE_VIEW_SETTINGS_KEY_PREFIX}${tableType}`,
                    value: JSON.stringify([tableSettings]),
                },
            })
            .finally(() => {
                setSaving(false);
            })
            .then(() => {
                onCloseModal(tableSettings?.columns);
            })
            .catch(setSavingError);
    };

    const content = (
        <>
            <div className={style.columns_container}>
                <div className={style.title}>{i18n('Add column')}</div>
                <div className={style.search_container}>
                    <Select
                        options={headers
                            .filter((headerItem) => {
                                let { key } = headerItem;

                                return !tableSettings?.columns.includes(key);
                            })
                            .map((headerItem) => {
                                let { key, displayName } = headerItem;

                                return { value: key, text: displayName };
                            })}
                        placeholder={i18n('Column name')}
                        onSelect={onAddColumn}
                        disabled={isLoading || isSaving || !!loadingError}
                    />
                </div>
                {loadingError ? (
                    <ErrorReloadLabel reloadFunction={getTableSettings} />
                ) : isLoading ? (
                    requiredColumns?.map((columnKey, index, arr) => {
                        return (
                            <React.Fragment key={columnKey}>
                                <div className={style.shimmer} />
                                {index !== arr.length - 1 ? <Separator /> : null}
                            </React.Fragment>
                        );
                    })
                ) : (
                    tableSettings?.columns &&
                    filterColumns(tableSettings?.columns)?.map((columnKey, index, arr) => {
                        let displayName =
                            headers.find((headerItem) => headerItem.key === columnKey)?.displayName ?? columnKey;

                        return (
                            <React.Fragment key={columnKey}>
                                <CustomizeTableItem
                                    displayName={displayName}
                                    onRemoveColumn={onRemoveColumn.bind(null, columnKey)}
                                    required={requiredColumns.includes(columnKey)}
                                />

                                {index !== arr.length - 1 ? <Separator /> : null}
                            </React.Fragment>
                        );
                    })
                )}
            </div>
            <div className={style.separator_container}>
                <Separator />
            </div>
            <div className={style.buttons_container}>
                {savingError ? (
                    <div className={style.error_container}>
                        <ErrorLabel simple />
                    </div>
                ) : null}
                <ControlButton
                    secondary
                    fullWidth
                    title={i18n('Reset settings')}
                    onClick={onResetSettings}
                    disabled={isLoading || isSaving || !!loadingError}
                />

                <ControlButton
                    fullWidth
                    title={i18n('Save')}
                    onClick={onSaveClick}
                    disabled={isLoading || isSaving || !!loadingError}
                />
            </div>
        </>
    );

    return (
        <Modal
            onClose={onCloseModal}
            title={i18n('Table customization')}
            contentComponent={content}
        />
    );
};

export default CustomizeTableModal;
