import { EMPTY_DATA } from 'constants/constants';

import { getMileageString } from 'utils/car/getMileageString';
import { ICarInfo, ICarScoring } from 'utils/car/types';
import { isValidJSONString } from 'utils/isValidJSONString';

import { filterCarsTableColumns } from 'entities/Car/helpers/filterCarsTableColumns/filterCarsTableColumns';
import { getCarSignalqStatus } from 'entities/Car/helpers/getCarSignalqStatus/getCarSignalqStatus';
import { getCarStatus } from 'entities/Car/helpers/getCarStatus/getCarStatus';
import { getCarSpeed } from 'entities/Car/helpers/getCarSpeed/getCarSpeed';

import { ORDER_DESC_CGI, ORDER_FIELD_CGI } from 'shared/consts/order';

import { GSMSignalStrength } from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/GSMSignalIcon';
import { TABLE_VIEW_SETTINGS_KEY_PREFIX } from 'components/Cars/CarsSidebar/CustomizeTableButton';
import { CARS_HEADER, DEFAULT_CARS_COLUMNS } from 'components/Cars/CarsTable/constants';
import { StatusTypes } from 'components/types';
import { REQUESTS } from 'components/ui/SectionLayout/SectionFiltersSidebar/MetaControlButton/MetaControlSave/request';
import { getNumberDisplayString } from 'components/ui/Table/getNumberDisplayString';
import { IHeaderInfo } from 'components/ui/Table/types';

import { ABORT_ERROR_KEY } from '../../../../request-helper/src';

import { i18n } from 'components/Cars/CarsTable/downloadCarTableXLSX.i18n';

const TRANSLATES = {
    Модель: i18n('Model'),
    Номер: i18n('Number'),
    Статус: i18n('Status'),
    SH: i18n('SH'),
    Сдаваемость: i18n('Utilization'),
    Таксопарк: i18n('Taxi park'),
    Лейблы: i18n('Labels'),
    'Текущая скорость': i18n('Current speed'),
    Топливо: i18n('Fuel'),
    Заряд: i18n('Battery charge'),
    'Темп. двигателя': i18n('Engine temp.'),
    'GPS сигнал': i18n('GPS signal'),
    'GPS используемых': i18n('GPS used'),
    'GPS видимых': i18n('GPS in view'),
    'GSM сигнал': i18n('GSM signal'),
    Пробег: i18n('Mileage'),
    'Текущее местоположение': i18n('Current Location'),
    Скоринг: i18n('Scoring'),
    'Запас хода': i18n('Range'),
};

const buildTable = (cars: Array<ICarInfo & ICarScoring>) => {
    return cars?.map((car) => {
        const {
            model = null,
            taxi_park = null,
            leasing_stats,
            status,
            signalq,
            signalq_status,
            telematics,
            location,
        } = car || {};

        const hasSignalq = Boolean(signalq);

        const { sh, utilization } = leasing_stats;

        const { lon, lat } = location ?? {};

        const {
            mileage,
            ext_voltage,
            fuel_level,
            fuel_distance,
            gsm_signal_level,
            VEGA_GPS_INVIEW,
            gps_used,
            engine_temperature,
        } = telematics ?? {};

        let { value: valueSh, type: valueTypeSh } = sh ?? {};
        let { value: valueUtilization, type: valueTypeUtilization } = utilization ?? {};
        let taxParkNameDisplay = taxi_park ?? EMPTY_DATA;

        let gsmValueSticks: number | null = null;
        if (gsm_signal_level !== null && gsm_signal_level !== undefined) {
            const LOW_SIGNAL_STICKS = 1;
            const MEDIUM_SIGNAL_STICKS = 2;
            const STRONG_SIGNAL_STICKS = 3;
            const HIGH_SIGNAL_STICKS = 4;

            if (gsm_signal_level === 0) {
                gsmValueSticks = 0;
            }

            if (gsm_signal_level >= GSMSignalStrength.LOW && gsm_signal_level < GSMSignalStrength.MEDIUM) {
                gsmValueSticks = LOW_SIGNAL_STICKS;
            }
            if (gsm_signal_level >= GSMSignalStrength.MEDIUM && gsm_signal_level < GSMSignalStrength.STRONG) {
                gsmValueSticks = MEDIUM_SIGNAL_STICKS;
            }
            if (gsm_signal_level >= GSMSignalStrength.STRONG && gsm_signal_level < GSMSignalStrength.STRONGEST) {
                gsmValueSticks = STRONG_SIGNAL_STICKS;
            }
            if (gsm_signal_level >= GSMSignalStrength.STRONGEST) {
                gsmValueSticks = HIGH_SIGNAL_STICKS;
            }
        }

        return {
            model: model ?? EMPTY_DATA,
            number: car?.number,
            status: getCarStatus(status),
            signalq_status: hasSignalq ? getCarSignalqStatus(signalq_status) : EMPTY_DATA,
            gsm_signal_level:
                status === StatusTypes.NO_SIGNAL
                    ? `0/4`
                    : gsm_signal_level !== null && gsm_signal_level !== undefined
                    ? `${gsmValueSticks}/4`
                    : null,
            gps_inview: status === StatusTypes.NO_SIGNAL ? 0 : VEGA_GPS_INVIEW ?? null,
            gps_used: status === StatusTypes.NO_SIGNAL ? 0 : gps_used,
            engine_temperature:
                engine_temperature !== null && engine_temperature !== undefined && status !== StatusTypes.NO_SIGNAL
                    ? `${Number(engine_temperature).toFixed(1)} °C`
                    : EMPTY_DATA,
            vin: car.vin,
            mileage: getMileageString(mileage ?? null),
            location: lon && lat && status !== StatusTypes.NO_SIGNAL ? [lon, lat].join(', ') : EMPTY_DATA,
            ext_voltage: ext_voltage && status !== StatusTypes.NO_SIGNAL ? `${ext_voltage.toFixed(1)} V` : EMPTY_DATA,
            speed: getCarSpeed(car),
            fuel_level:
                fuel_level !== null &&
                fuel_level !== undefined &&
                status !== StatusTypes.NO_SIGNAL &&
                typeof fuel_level === 'number'
                    ? `${fuel_level.toFixed(1)} %`
                    : EMPTY_DATA,
            fuel_distance:
                status !== StatusTypes.NO_SIGNAL && fuel_distance ? `${fuel_distance} ${i18n('km')}` : EMPTY_DATA,
            scoring: car?.aggressive_rank ?? EMPTY_DATA,
            sh: getNumberDisplayString(valueSh, valueTypeSh),
            utilization: getNumberDisplayString(valueUtilization, valueTypeUtilization),
            taxi_park: taxParkNameDisplay,
        };
    });
};

export const downloadCarTableXLSX = async (request, abortGettingCars, getCars, location) => {
    try {
        let headerColumns = await request.exec(REQUESTS.GET_SETTINGS).then((settingsArray) => {
            let tableViewCarsSettings =
                settingsArray?.settings?.find(
                    (settingsItem) => settingsItem.id === `${TABLE_VIEW_SETTINGS_KEY_PREFIX}cars`,
                ) ?? [];
            let tableSettings = isValidJSONString(tableViewCarsSettings.value)
                ? JSON.parse(tableViewCarsSettings.value)?.[0]
                : null;

            return {
                columns: tableSettings?.columns ?? DEFAULT_CARS_COLUMNS,
            };
        });
        abortGettingCars();

        let cars: any[] = [];
        let canGetMore = true;
        let page = 1;

        let urlSearchParams = new URLSearchParams(location?.search);
        let orderField = urlSearchParams.get(ORDER_FIELD_CGI);
        let orderDesc = urlSearchParams.get(ORDER_DESC_CGI);
        let sort = orderField ? 'leasing_stats' : null;

        do {
            let response = await getCars({
                page,
                order_field: orderField,
                order_desc: orderDesc,
                sort,
                page_size: 1000,
            }).then((response) => {
                if (response.meta !== ABORT_ERROR_KEY) {
                    return response;
                }
            });
            if (response) {
                canGetMore = response.canGetMore;
                cars.push(...(response.cars ?? []));
                page = response.currentPage + 1;
            }
        } while (canGetMore);

        let headers: IHeaderInfo[] = CARS_HEADER.reduce((memo, header) => {
            if (headerColumns?.columns.includes(header.key)) {
                if (header.key === 'gps_value') {
                    memo = memo.concat([
                        { key: 'gps_used', displayName: 'GPS используемых' },
                        { key: 'gps_inview', displayName: 'GPS видимых' },
                    ]);
                } else {
                    memo.push(header);
                }
            }

            return memo;
        }, [] as IHeaderInfo[]);

        headers = filterCarsTableColumns<IHeaderInfo>(headers);

        headers = headers.map((header) => {
            header.displayName = TRANSLATES[header.displayName as any] || header.displayName;

            return header;
        });

        return { headers, table: buildTable(cars), name: `cars_${Date.now()}` };
    } catch (e) {
        throw e;
    }
};
