import React, { MutableRefObject, useEffect, useRef, useState } from 'react';
import { useHistory } from 'react-router-dom';

import { EMPTY_DATA } from 'constants/constants';

import { getMileageString } from 'utils/car/getMileageString';
import { ICarInfo, ICarScoring } from 'utils/car/types';
import { isValidJSONString } from 'utils/isValidJSONString';
import { SupportedLanguage } from 'utils/language/types';

import { CarsContentHeader } from 'widgets/CarsContentHeader';

import { CarCameraStatus } from 'features/CarCameraStatus';

import { filterCarsTableColumns } from 'entities/Car/helpers/filterCarsTableColumns/filterCarsTableColumns';
import { getCarSpeed } from 'entities/Car/helpers/getCarSpeed/getCarSpeed';

import { GPSValueIcon } from 'shared/ui/GPSValueIcon/GPSValueIcon';
import { Scoring } from 'shared/ui/Scoring/Scoring';
import { Widget } from 'shared/ui/Widget/Widget';

import GSMSignalIcon, { GSMSignalStrength } from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/GSMSignalIcon';
import { CarNumberLabel } from 'components/Cars/CarNumberLabel';
import { TABLE_VIEW_SETTINGS_KEY_PREFIX } from 'components/Cars/CarsSidebar/CustomizeTableButton';
import { CARS_HEADER, DEFAULT_CARS_COLUMNS } from 'components/Cars/CarsTable/constants';
import CarStatus from 'components/Cars/CarStatus';
import { IWithCarRequest } from 'components/Cars/index';
import { REQUESTS, SETTINGS_REQUESTS } from 'components/Reports/request';
import { StatusTypes } from 'components/types';
import Table from 'components/ui/Table';
import { getNumberDisplayString } from 'components/ui/Table/getNumberDisplayString';
import { IHeaderInfo, ITableRowData } from 'components/ui/Table/types';

import { ABORT_ERROR_KEY, RequestHelper } from '../../../../request-helper/src';

import { i18n } from 'components/Cars/CarsTable/index.i18n';

import styles from 'components/Cars/CarsTable/index.css';

export interface ILangProps {
    lang?: SupportedLanguage;
}

interface ICarsTableProps extends IWithCarRequest {
    carsTableColumns?: string[];
    tagsFilter: string;

    onCarsTableColumnsChange(columns: string[]): {};
}

const CarsTable = ({
    filters,
    getCars: getCarsProps,
    abortGettingCars,
    carsTableColumns,
    tagsFilter,
    onCarsTableColumnsChange,
}: ICarsTableProps) => {
    const [currentPage, setCurrentPage] = useState<number>(1);
    const [canGetMore, setCanGetMore] = useState<boolean>(false);
    const [isLoading, setIsLoading] = useState<boolean>(true);
    const [isMoreLoading, setIsMoreLoading] = useState<boolean>(true);
    const [error, setError] = useState<Error | null>(null);
    const [cars, setCars] = useState<ICarInfo[]>([]);
    //if null, no sorting
    const [sorting, setSorting] = useState<{ order_field: string | null; order_desc: boolean | null } | null>(null);
    const [isTableSettingsLoading, setTableSettingsIsLoading] = useState<boolean>(true);
    const [tableSettings, setTableSettings] = useState<{ columns: string[] } | null>(null);
    const [header, setHeader] = useState<IHeaderInfo[]>([]);
    const [tableRows, setTableRows] = useState<ITableRowData[]>([]);

    const request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: SETTINGS_REQUESTS }));

    let history = useHistory();

    useEffect(() => {
        getCars(currentPage, false);
        getCarsTableSettings();
    }, []);

    useEffect(() => {
        getCars(1, false);
        getCarsTableSettings();
    }, [
        filters.telematics,
        filters.camera,
        filters.service,
        filters.no_sh,
        filters.park_id,
        filters.status,
        filters.signalq_status,
        filters.fresh_issue_date,
        sorting,
        carsTableColumns,
        // Place for custom filter
        filters.class,
    ]);

    const getCars = (page: number, isMore = false) => {
        if (isMore) {
            setIsMoreLoading(true);
        } else {
            setIsLoading(true);
        }
        setError(null);
        abortGettingCars();

        let options = {
            page,
            sort: sorting?.order_field ? 'leasing_stats' : null,
            order_field: sorting?.order_field ?? null,
            order_desc: sorting?.order_desc ?? null,
        };

        getCarsProps(options)
            .then((response) => {
                if (response.meta !== ABORT_ERROR_KEY) {
                    let { cars: carsResponse, canGetMore, currentPage } = response;

                    setCars(isMore ? [...cars, ...carsResponse] : carsResponse);
                    setTableRows(buildTableRows(isMore ? [...cars, ...carsResponse] : carsResponse));
                    setCanGetMore(canGetMore);
                    setCanGetMore(canGetMore);
                    setCurrentPage(currentPage);

                    setIsMoreLoading(false);
                    setIsLoading(false);
                }
            })
            .catch((error) => {
                setError(error);

                setIsMoreLoading(false);
                setIsLoading(false);
            });
    };

    const getCarsTableSettings = () => {
        setTableSettingsIsLoading(true);
        request.current.exec(REQUESTS.GET_SETTINGS).then((settingsArray) => {
            let tableViewCarsSettings =
                settingsArray?.settings?.find(
                    (settingsItem) => settingsItem.id === `${TABLE_VIEW_SETTINGS_KEY_PREFIX}cars`,
                ) ?? [];
            let tableSettings = isValidJSONString(tableViewCarsSettings.value)
                ? JSON.parse(tableViewCarsSettings.value)?.[0]
                : null;

            setTableSettings({
                columns: tableSettings?.columns ?? DEFAULT_CARS_COLUMNS,
            });

            setTableSettingsIsLoading(false);
        });
    };

    useEffect(() => {
        let headers = tableSettings?.columns?.length
            ? CARS_HEADER.filter((carsHeaderItem) => tableSettings?.columns.includes(carsHeaderItem.key))
            : [];

        headers = filterCarsTableColumns<IHeaderInfo>(headers);

        setHeader(headers);
    }, [tableSettings]);

    const onGetMoreClick = () => {
        getCars(currentPage + 1, true);
    };

    const buildTableRows = (cars: Array<ICarInfo & ICarScoring>): ITableRowData[] => {
        return cars?.map((car, index) => {
            const onCarRowClick = () => {
                history.push(`/cars/${id}`);
            };

            const {
                id = null,
                model = null,
                taxi_park = null,
                leasing_stats,
                status,
                signalq,
                signalq_status,
                telematics,
                location,
            } = car || {};

            const hasSignalq = Boolean(signalq && signalq_status);

            const { sh, utilization } = leasing_stats;

            const { lon, lat } = location ?? {};

            const {
                mileage,
                ext_voltage,
                fuel_level,
                fuel_distance,
                gsm_signal_level,
                VEGA_GPS_INVIEW,
                gps_used,
                engine_temperature,
            } = telematics ?? {};

            const speed = getCarSpeed(car);

            const GPSInview = status === StatusTypes.NO_SIGNAL ? 0 : VEGA_GPS_INVIEW ?? null;
            const GPSUsed = status === StatusTypes.NO_SIGNAL ? 0 : gps_used ?? null;

            let { value: valueSh, type: valueTypeSh } = sh ?? {};
            let { value: valueUtilization, type: valueTypeUtilization } = utilization ?? {};
            let taxParkNameDisplay = taxi_park ?? EMPTY_DATA;

            let data = {
                model: model ?? EMPTY_DATA,
                number: (
                    <CarNumberLabel
                        carInfo={car}
                        shortVIN
                        onlyText
                    />
                ),

                status: (
                    <CarStatus
                        carInfo={car}
                        big
                        oneRow
                    />
                ),

                signalq_status: hasSignalq ? <CarCameraStatus status={signalq_status!} /> : <span>{EMPTY_DATA}</span>,

                gsm_signal_level: (
                    <GSMSignalIcon
                        GSMValue={status === StatusTypes.NO_SIGNAL ? GSMSignalStrength.ZERO : gsm_signal_level ?? null}
                    />
                ),

                gps_value: (
                    <GPSValueIcon
                        GPSUsed={GPSUsed}
                        GPSInview={GPSInview}
                        short
                    />
                ),

                engine_temperature:
                    engine_temperature !== null && engine_temperature !== undefined && status !== StatusTypes.NO_SIGNAL
                        ? `${Number(engine_temperature).toFixed(1)} °C`
                        : EMPTY_DATA,
                vin: (
                    <CarNumberLabel
                        carInfo={{ vin: car.vin }}
                        onlyText
                    />
                ),

                mileage: getMileageString(mileage ?? null),
                location: lon && lat && status !== StatusTypes.NO_SIGNAL ? [lon, lat].join(', ') : EMPTY_DATA,
                ext_voltage:
                    ext_voltage && status !== StatusTypes.NO_SIGNAL ? `${ext_voltage.toFixed(1)} V` : EMPTY_DATA,
                speed: speed,
                fuel_level:
                    fuel_level !== null &&
                    fuel_level !== undefined &&
                    typeof fuel_level === 'number' &&
                    status !== StatusTypes.NO_SIGNAL
                        ? `${fuel_level.toFixed(1)} %`
                        : EMPTY_DATA,
                fuel_distance:
                    status !== StatusTypes.NO_SIGNAL && fuel_distance ? `${fuel_distance} ${i18n('km')}` : EMPTY_DATA,
                sh: getNumberDisplayString(valueSh, valueTypeSh),
                utilization: getNumberDisplayString(valueUtilization, valueTypeUtilization),
                scoring: <Scoring value={car?.aggressive_rank} />,
                taxi_park: (
                    <div
                        className={styles.taxi_park_name_td}
                        title={taxi_park ?? EMPTY_DATA}
                    >
                        <div className={styles.taxi_park_name}>{taxParkNameDisplay}</div>
                    </div>
                ),
            };

            return {
                data,
                meta: {
                    key: car.vin || car.number || index.toString(),
                    className: styles.car_row,
                    onClick: onCarRowClick,
                },
            };
        });
    };

    const onSortChange = (key: string, sortDesc: boolean) => {
        if (key) {
            setSorting({ order_field: key, order_desc: sortDesc });
        } else {
            setSorting(null);
        }
    };

    return (
        <Widget contentContainer>
            <CarsContentHeader
                tagsFilter={tagsFilter}
                filters={filters}
                getData={getCarsProps}
                abortGettingCars={abortGettingCars}
                onCarsTableColumnsChange={onCarsTableColumnsChange}
            />

            <Table
                isLoading={isLoading || isTableSettingsLoading}
                error={error}
                getMore={{
                    canGetMore,
                    onGetMoreClick,
                    isMoreLoading,
                }}
                header={header}
                tableData={tableRows}
                reloadFunction={getCars.bind(null, 1, false, true)}
                onSortChange={onSortChange}
            />
        </Widget>
    );
};

export default CarsTable;
