import React, { useEffect, useState } from 'react';
import { RouteComponentProps, useHistory, useLocation, withRouter } from 'react-router-dom';

import { DEBOUNCE_TIMEOUT, MAP_DEFAULT_LAT, MAP_DEFAULT_LON, MAP_DEFAULT_ZOOM } from 'constants/constants';

import { ICarInfo } from 'utils/car/types';
import { getAreaTags } from 'utils/getAreaTags';
import { getCenter } from 'utils/getCenter';
import { IAreaItem } from 'utils/map/area/types';
import getCgiMapOptions from 'utils/map/map/getCgiMapOptions';

import { IAreasFilters, IWithCarRequest } from 'components/Cars/index';
import { CARS_REQUESTS, REQUESTS } from 'components/Cars/request';
import GlobalErrorHandler from 'components/GlobalErrorHandler';
import { AreasButton } from 'components/ui/Buttons/AreasButton';
import { Areas } from 'components/ui/Buttons/AreasButton/AreasWidget';

import { RequestHelper } from '../../../../request-helper/src';

import style from 'components/Cars/CarsView/index.css';

type Timeout = NodeJS.Timeout;

const DEFAULT_ZOOM = MAP_DEFAULT_ZOOM;

interface IMapOptions {
    zoom: number;
    center: [number, number]; //[lon, lat]
}

interface ICarsViewProps extends RouteComponentProps, IWithCarRequest {}

interface ICarsViewState {
    mapOptions: IMapOptions;
    cars: ICarInfo[];
    areas: IAreaItem[];
    areasFilters: IAreasFilters;
}

const CarsMap = React.lazy(() => import('components/Map/CarsMap'));

class CarsView extends React.Component<ICarsViewProps, ICarsViewState> {
    state: ICarsViewState = {
        mapOptions: {
            zoom: MAP_DEFAULT_ZOOM,
            center: [MAP_DEFAULT_LON, MAP_DEFAULT_LAT],
        },
        cars: [],
        areas: [],
        areasFilters: {
            [Areas.FORBIDDEN_TO_RIDE]: false,
            [Areas.FORBIDDEN_TO_PARK]: false,
        },
    };
    request = new RequestHelper({ requestConfigs: CARS_REQUESTS });
    debounceTimeout: Timeout;

    constructor(props: ICarsViewProps) {
        super(props);
        let { zoom, lon, lat } = getCgiMapOptions(this.props);
        let center = getCenter();

        this.state = {
            mapOptions: {
                zoom: zoom !== null && !isNaN(+zoom) ? +zoom : DEFAULT_ZOOM,
                center: [
                    lon !== null && !isNaN(+lon) ? +lon : center[0],
                    lat !== null && !isNaN(+lat) ? +lat : center[1],
                ],
            },
            cars: [],
            areas: [],
            areasFilters: {
                [Areas.FORBIDDEN_TO_RIDE]: false,
                [Areas.FORBIDDEN_TO_PARK]: false,
            },
        };
    }

    componentDidMount() {
        this.getMapCars();
        this.getMapAreas();
    }

    componentWillUnmount() {
        this.request.abort();
        this.props.abortGettingCars();
        clearTimeout(this.debounceTimeout);
    }

    componentDidUpdate(prevProps) {
        if (
            this.props.filters.telematics !== prevProps.filters.telematics ||
            this.props.filters.camera !== prevProps.filters.camera ||
            this.props.filters.service !== prevProps.filters.service ||
            this.props.filters.no_sh !== prevProps.filters.no_sh ||
            this.props.filters.park_id !== prevProps.filters.park_id ||
            this.props.filters.status !== prevProps.filters.status ||
            this.props.filters.signalq_status !== prevProps.filters.signalq_status ||
            this.props.filters.fresh_issue_date !== prevProps.filters.fresh_issue_date ||
            // Place for custom filter
            this.props.filters.class !== prevProps.filters.class
        ) {
            this.getMapCars();
        }

        if (
            this.props.filters?.[Areas.FORBIDDEN_TO_RIDE] !== prevProps.filters?.[Areas.FORBIDDEN_TO_RIDE] ||
            this.props.filters?.[Areas.FORBIDDEN_TO_PARK] !== prevProps.filters?.[Areas.FORBIDDEN_TO_PARK]
        ) {
            this.getMapAreas();
        }
    }

    getMapCars() {
        this.props
            .getCars()
            .then((response) => {
                let { cars } = response;
                this.setState({ cars });
            })
            .catch((error) => {
                console.error(error);
            });
    }

    getMapAreas() {
        let area_tags = getAreaTags();
        area_tags &&
            this.request.exec(REQUESTS.AREAS_INFO).then((response) => {
                this.setState({
                    areas: response.areas?.filter((area) => area.area_tags.includes(area_tags)),
                });
            });
    }

    onAreasFilterChange(areasFilters: IAreasFilters) {
        clearTimeout(this.debounceTimeout);
        this.debounceTimeout = setTimeout(() => {
            this.setState({ areasFilters });
        }, DEBOUNCE_TIMEOUT);
    }

    render() {
        let { mapOptions, cars } = this.state;

        return (
            <div className={style.map_container}>
                <GlobalErrorHandler children={null} />
                <React.Suspense fallback={<div />}>
                    <CarsMap
                        options={{
                            zoom: mapOptions.zoom,
                            center: mapOptions.center,
                        }}
                        cars={cars}
                        areas={this.state.areas}
                        showAreas={this.state.areasFilters}
                    />
                </React.Suspense>
                <div className={style.controls}>
                    <CarsAreasControl onAreasFilterChange={this.onAreasFilterChange.bind(this)} />
                </div>
            </div>
        );
    }
}

export default withRouter<ICarsViewProps, any>(CarsView);

interface ICarsAreasControlProps {
    onAreasFilterChange: (filters: IAreasFilters) => {};
}

export const CarsAreasControl = (props: ICarsAreasControlProps) => {
    const [areasFilters, setAreasFilters] = useState<IAreasFilters>({
        [Areas.FORBIDDEN_TO_RIDE]: false,
        [Areas.FORBIDDEN_TO_PARK]: false,
    });
    let history = useHistory();
    let location = useLocation();
    let areaTags = getAreaTags();

    useEffect(() => {
        let searchParams = new URLSearchParams(location.search);

        let areasFilters: IAreasFilters = {
            [Areas.FORBIDDEN_TO_RIDE]: false,
            [Areas.FORBIDDEN_TO_PARK]: false,
        };

        [Areas.FORBIDDEN_TO_RIDE, Areas.FORBIDDEN_TO_PARK].forEach((filterSearchKey) => {
            if (searchParams.get(filterSearchKey) === 'true') {
                areasFilters[filterSearchKey] = true;
            }
            if (searchParams.get(filterSearchKey) === 'false' || searchParams.get(filterSearchKey) === null) {
                areasFilters[filterSearchKey] = false;
            }
        });

        setAreasFilters(areasFilters);
    }, [location]);

    useEffect(() => {
        props.onAreasFilterChange(areasFilters);
    }, [areasFilters]);

    const onFilterChange = (type: string | string[], value: string | boolean | string[] | boolean[]) => {
        let search = location?.search;
        let urlSearchParams = new URLSearchParams(search);

        const setSearchParam = (type: string, value: string) => {
            if (value !== null) {
                urlSearchParams.set(type, value);
            } else {
                urlSearchParams.delete(type);
            }
        };

        if (Array.isArray(type) && Array.isArray(value)) {
            type.forEach((typeItem, index) => {
                setSearchParam(typeItem, value[index].toString());
            });
        } else {
            setSearchParam(type as string, value.toString());
        }

        history.push(`${location.pathname}?${urlSearchParams}`);
    };

    const onAreasCheck = (isForbiddenToRide: boolean, isForbiddenToPark: boolean) => {
        onFilterChange([Areas.FORBIDDEN_TO_RIDE, Areas.FORBIDDEN_TO_PARK], [isForbiddenToRide, isForbiddenToPark]);
    };

    let checked = {
        [Areas.FORBIDDEN_TO_RIDE]: areasFilters?.[Areas.FORBIDDEN_TO_RIDE],
        [Areas.FORBIDDEN_TO_PARK]: areasFilters?.[Areas.FORBIDDEN_TO_PARK],
    };

    return areaTags ? (
        <AreasButton
            checked={checked}
            onCheck={onAreasCheck}
        />
    ) : null;
};
