import React from 'react';
import { RouteComponentProps, withRouter } from 'react-router-dom';

import { CarsFilters } from 'widgets/CarsFilters';
import { Header, HEADER_HEIGHT } from 'widgets/Header';

import { CarsHeaderTabs } from 'features/CarsHeaderTabs';

import { CarsFilter } from 'entities/Car/consts/CarsFilter';
import { CarsView as CarsViews } from 'entities/Car/consts/CarsView';
import { INITIAL_CARS_FILTERS } from 'entities/Car/consts/filters';

import { ContentContainer } from 'shared/ui/ContentContainer/ContentContainer';
import { SectionLayout } from 'shared/ui/SectionLayout/SectionLayout';

import CarsTable from 'components/Cars/CarsTable';
import CarsView from 'components/Cars/CarsView';
import { SERVICE_TAG } from 'components/Cars/constants';
import { CARS_REQUESTS, REQUESTS } from 'components/Cars/request';
import { Areas } from 'components/ui/Buttons/AreasButton/AreasWidget';

import { ABORT_ERROR_KEY, RequestHelper } from '../../../request-helper/src';

import { i18n } from 'components/Cars/index.i18n';

type Timeout = NodeJS.Timeout;

const SENSORS = [
    'speed',
    'mileage',
    'fuel_level',
    'fuel_distance',
    'ext_voltage',
    'engine_on',
    'gsm_signal_level',
    'VEGA_GPS_INVIEW',
    'gps_used',
    'engine_temperature',
].join(',');

const PAGE_SIZE = 50;

export interface CarsFiltersOptions {
    // Custom for CarLove received via 'carlove_custom_filters' flag
    [CarsFilter.CLASS]: Nullable<string>;

    [CarsFilter.NO_SH]: Nullable<boolean>;
    [CarsFilter.TELEMATICS]: Nullable<boolean>;
    [CarsFilter.CAMERA]: Nullable<boolean>;
    [CarsFilter.FRESH_ISSUE_DATE]: Nullable<string>;
    [CarsFilter.PARK_ID]: Nullable<string>;
    [CarsFilter.SERVICE]: Nullable<boolean>;
    [CarsFilter.STATUS]: Nullable<string>;
    [CarsFilter.SIGNALQ_STATUS]: Nullable<string>;
}

export interface IAreasFilters {
    [Areas.FORBIDDEN_TO_RIDE]: boolean;
    [Areas.FORBIDDEN_TO_PARK]: boolean;
}

export interface IWithCarRequest {
    filters: CarsFiltersOptions;

    getCars(options?: any): Promise<any>;

    abortGettingCars(): void;
}

interface ICarsState {
    filters: CarsFiltersOptions;
    tagsFilter: string;
    carsView: Nullable<CarsViews>;
    carsTableColumns: string[];
}

class Cars extends React.Component<RouteComponentProps, ICarsState> {
    state: ICarsState = {
        filters: INITIAL_CARS_FILTERS,
        tagsFilter: '',
        carsView: null,
        carsTableColumns: [],
    };

    request = new RequestHelper({ requestConfigs: CARS_REQUESTS });
    debounceTimeout: Timeout;

    componentWillUnmount() {
        this.abortGettingCars();
        clearTimeout(this.debounceTimeout);
    }

    getCars(options: any) {
        let { filters } = this.state;
        let { no_sh, fresh_issue_date, status, signalq_status, telematics, camera, service } = filters;

        let searchParams = new URLSearchParams(this.props.location?.search);
        let park_id = searchParams.get('park_id');
        let parksIds = park_id ? park_id.split(',').filter((parkId) => parkId !== 'all') : [];

        // Place for custom filter
        let car_class = searchParams.get('class');
        let carClassTags = car_class ? car_class.split(',').filter((_class) => _class !== 'all') : [];

        let { page, sort, order_field, order_desc, page_size } = options ?? {};
        let tagsFilterSingle = [
            no_sh ? 'insufficient_sh_tag' : '',
            telematics ? 'leasing_has_telematics_tag' : '',
            camera ? 'leasing_has_signalq_tag' : '',
            // Place for custom filter
            carClassTags ? carClassTags.join('*') : '',

            service ? SERVICE_TAG : '',
            !fresh_issue_date || fresh_issue_date === 'all' ? '' : fresh_issue_date,
        ]
            .filter(Boolean)
            .join('*');

        let tagsFilter = parksIds.length
            ? parksIds.map((parkId) => `${parkId}*${tagsFilterSingle}`).join(',')
            : tagsFilterSingle;

        let queryParams: any = {
            tags_filter: tagsFilter,
            sensors: SENSORS,
            traits: 'ReportVIN,ReportAggressiveScore,ReportTagDetails,ReportCommonSpeed,ReportSignalq',
            status: status ? status : null,
            signalq_status_filter: signalq_status ? signalq_status : null,
        };

        if (page !== null && page !== undefined) {
            queryParams.page_number = page;
            queryParams.page_size = page_size ?? PAGE_SIZE;
        }
        if (sort !== null && sort !== undefined) {
            queryParams.sort = sort;
            queryParams.order_field = order_field;
            queryParams.order_desc = order_desc.toString();
        }

        //Save tagsFilter string for saving this value in group
        this.setState({ tagsFilter });

        this.abortGettingCars();

        return this.request
            .exec(REQUESTS.GET_CARS, { queryParams })
            .then((response) => {
                if (response.meta !== ABORT_ERROR_KEY) {
                    let { cars, models, can_get_more_pages } = response;

                    let carsData = cars.map((car) => {
                        let carCopy = Object.assign({}, car);
                        carCopy.model = models?.[carCopy?.model_id]?.name || carCopy?.model_id;
                        carCopy.number = carCopy.number?.toUpperCase();
                        carCopy.taxi_park = carCopy?.taxi_company?.name;

                        return carCopy;
                    });

                    return {
                        cars: carsData,
                        canGetMore: can_get_more_pages,
                        currentPage: page,
                    };
                }

                return response;
            })
            .catch((error) => {
                throw error;
            });
    }

    abortGettingCars() {
        this.request.abort();
    }

    onCarsViewChange(carsView: CarsViews) {
        this.setState({ carsView });
    }

    onFiltersChange(filters: CarsFiltersOptions) {
        this.setState({ filters });
    }

    onCarsTableColumnsChange(columns: string[]) {
        this.setState({ carsTableColumns: columns });
    }

    render() {
        let { carsView, filters, carsTableColumns, tagsFilter } = this.state;

        let contentComponent: React.ClassicElement<any> | null = null;

        switch (carsView) {
            case CarsViews.TABLE:
                contentComponent = (
                    <ContentContainer
                        bodyScroll
                        withSidebar
                    >
                        <CarsTable
                            filters={filters}
                            tagsFilter={tagsFilter}
                            carsTableColumns={carsTableColumns}
                            getCars={this.getCars.bind(this)}
                            abortGettingCars={this.abortGettingCars.bind(this)}
                            onCarsTableColumnsChange={this.onCarsTableColumnsChange.bind(this)}
                        />
                    </ContentContainer>
                );

                break;
            case CarsViews.MAP:
                contentComponent = (
                    <CarsView
                        filters={filters}
                        getCars={this.getCars.bind(this)}
                        abortGettingCars={this.abortGettingCars.bind(this)}
                    />
                );

                break;
        }

        return (
            <SectionLayout
                header={
                    <Header
                        title={i18n('Cars')}
                        tabs={<CarsHeaderTabs onChangeCarsView={this.onCarsViewChange.bind(this)} />}
                        withoutBorder
                    />
                }
                filters={
                    <CarsFilters
                        offsetTop={HEADER_HEIGHT}
                        onFiltersChange={this.onFiltersChange.bind(this)}
                    />
                }
                bodyScroll
            >
                {contentComponent}
            </SectionLayout>
        );
    }
}

export default withRouter<any, any>(Cars);
