import React, { lazy, MutableRefObject, Suspense, useEffect, useReducer, useRef, useState } from 'react';
import { BrowserRouter as Router, Redirect, Route, RouteComponentProps, Switch } from 'react-router-dom';

import { LS_FLAGS } from 'constants/constants';

import FlagsContext from 'contexts/FlagsContext';
import GlobalTooltipContext from 'contexts/GlobalTooltipContext';

import { CarDashboardPage, getCarDashboardFlag } from 'utils/getCarDashboardFlag';
import { getDashboardComponent } from 'utils/getDashboardComponent';
import { isCarOverviewFlag } from 'utils/isCarOverviewFlag';
import { isCarsGroupsAccess } from 'utils/isCarsGroupsAccess';
import { isSettingsPricingFlag } from 'utils/isSettingsPricingFlag';
import { isShowNotificationPageFlag } from 'utils/isShowNotificationPage';
import { isShowUsersRides } from 'utils/isShowUsersRides';
import { isShowUsersSection } from 'utils/isShowUsersSection';
import { isSignalsRedesignFlag } from 'utils/isSignalsRedesignFlag';
import { isUserDocumentsPageFlag } from 'utils/isUserDocumentsPageFlag';
import { isUserProfilePageFlag } from 'utils/isUserProfilePageFlag';
import { isBaseLang } from 'utils/language/isBaseLang';
import { showCarsSchedule } from 'utils/showCarsSchedule';
import { showTaxiCompanyRides } from 'utils/showTaxiCompanyRides';
import { showZones } from 'utils/showZones';

import { CurrentUserSchema } from 'entities/User/types/CurrentUserSchema';

import { Path } from 'shared/consts/Path';
import { initErrorLogger } from 'shared/helpers/errorLogger/errorLogger';
import { isVlootkit } from 'shared/helpers/isVlootkit/isVlootkit';
import { CacheRequest, CacheRequestContext } from 'shared/hooks/useCacheRequestContext/useCacheRequestContext';
import { CurrentUserContext } from 'shared/hooks/useCurrentUser/useCurrentUser';
import { PageLoading } from 'shared/ui/PageLoading/PageLoading';
import { RouterScrollTop } from 'shared/ui/RouterScrollTop/RouterScrollTop';

import { CONTENT_REQUESTS, REQUESTS } from 'components/Content/request';
import { IPassportUserInfo } from 'components/Content/types';
import GlobalSidebar from 'components/GlobalSidebar';
import { NotificationCenter } from 'components/NotificationCenter';
import NotificationCenterContext, {
    initNotificationsStore,
    NotificationReducer,
} from 'components/NotificationCenter/store';
import { SideMenu } from 'components/SideMenu';
import GlobalTooltip from 'components/ui/GlobalTooltip';
import { ITooltipProps } from 'components/ui/Tooltip';

import { RequestHelper } from '../../../request-helper/src';

import style from 'components/Content/index.css';

const Dashboard = lazy(() => import('pages/Dashboard/Dashboard'));
const DashboardRental = lazy(() => import('pages/Dashboard/DashboardRental'));
const Portfolio = lazy(() => import('components/Portfolio'));
const Cars = lazy(() => import('components/Cars'));
const CarsGroups = lazy(() => import('components/Cars/CarsGroups'));
const CarsSchedule = lazy(() => import('pages/Cars/Schedule/'));
const CarOverview = lazy(() => import('pages/Car/Overview/'));
const CarSignals = lazy(() => import('pages/Car/Signals/'));
const CarDamages = lazy(() => import('pages/Car/Damages/'));
const CarRedirect = lazy(() => import('pages/Car/Redirect/'));
const CarCard = lazy(() => import('components/Cars/CarCard'));
const SignalsPage = lazy(() => import('pages/Signals/Signals'));
const Signals = lazy(() => import('components/Signals'));
const ManageSignals = lazy(() => import('components/Signals/ManageSignals'));
const Zones = lazy(() => import('components/Zones'));
const Reports = lazy(() => import('components/Reports'));
const Users = lazy(() => import('components/Users'));
const UserProfile = lazy(() => import('pages/User/Profile'));
const UserDocuments = lazy(() => import('pages/User/Documents'));
const UserRides = lazy(() => import('pages/User/Rides'));
const Settings = lazy(() => import('components/Settings'));
const SettingsPricing = lazy(() => import('pages/Settings/Pricing'));
const SettingsNotifications = lazy(() => import('pages/Settings/Notifications'));
const ErrorNotFoundView = lazy(() => import('components/ErrorView/404'));
const ErrorPermissionsDenyView = lazy(() => import('components/ErrorView/403'));
const Rides = lazy(() => import('components/Rides'));

interface IRoute {
    path: Path;
    exact?: boolean;
    strict?: boolean;
    render?: (props: RouteComponentProps<any>, withErrorHandler: (props) => JSX.Element) => React.ReactNode;
    redirect?: string;
    redirectFrom?: string;
    component: React.ExoticComponent;
}

const Content = () => {
    const [isLoading, setIsLoading] = useState<boolean>(true);
    const [userInfo, setUserInfo] = useState<IPassportUserInfo | null>(null);
    const [currentUser, setCurrentUser] = useState<CurrentUserSchema | null>(null);
    const [routes, setRoutes] = useState<IRoute[]>([]);
    const [flags, setFlags] = useState<{ [keys: string]: any } | null>(null);

    const [show, setShow] = useState(false);
    const [tooltipOptions, setTooltipOptions] = useState<ITooltipProps | null>(null);
    const [elementPosition, setElementPosition] = useState<[number | null, number | null]>([null, null]);
    const [elementSize, setElementSize] = useState<[number | null, number | null]>([null, null]);

    const request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: CONTENT_REQUESTS }));

    const [notificationsStore, notificationDispatch] = useReducer(NotificationReducer, initNotificationsStore);

    useEffect(() => {
        Promise.all([
            request.current.exec(REQUESTS.GET_USER_INFO, { queryParams: {} }),
            request.current.exec(REQUESTS.GET_FLAGS),
        ])
            .then((response) => {
                let [userResponse, flagsResponse] = response;

                initErrorLogger(userResponse?.default_uid);
                setUserInfo(userResponse);

                let flags = flagsResponse?.flags ?? {};
                let currentUser = flagsResponse?.user ?? {};

                setFlags(flags);
                setCurrentUser(currentUser);
                localStorage.setItem(LS_FLAGS, JSON.stringify(flags));
                setIsLoading(false);
            })
            .catch(() => {
                initErrorLogger(undefined);
            });

        return () => {
            request.current.abort();
        };
    }, []);

    useEffect(() => {
        let routes: IRoute[] = [
            ...(getCarDashboardFlag() === CarDashboardPage.FLEET
                ? [{ path: Path.DASHBOARD, exact: true, component: Dashboard }]
                : []),
            ...(getCarDashboardFlag() === CarDashboardPage.RENTAL
                ? [{ path: Path.DASHBOARD, exact: true, component: DashboardRental }]
                : []),
            { path: Path.CARS, exact: true, component: Cars },
            ...(isCarsGroupsAccess() ? [{ path: Path.CARS_GROUPS, exact: true, component: CarsGroups }] : []),
            ...(showCarsSchedule() ? [{ path: Path.CARS_SCHEDULE, exact: true, component: CarsSchedule }] : []),
            { path: Path.CAR_OVERVIEW, exact: true, component: isCarOverviewFlag() ? CarOverview : CarCard },
            { path: Path.CAR_SIGNALS, exact: true, component: isSignalsRedesignFlag() ? CarSignals : CarCard },
            { path: Path.CAR_RIDES, exact: true, component: CarCard },
            { path: Path.CAR_TELEMATICS, exact: true, component: CarCard },
            { path: Path.CAR_DAMAGES, exact: true, component: CarDamages },
            {
                path: Path.CAR,
                exact: true,
                component: CarRedirect,
            },
            { path: Path.SIGNALS, exact: true, component: isSignalsRedesignFlag() ? SignalsPage : Signals },
            { path: Path.SIGNALS_MANAGE, exact: true, component: ManageSignals },
            ...(showZones() ? [{ path: Path.SIGNALS_ZONES, exact: true, component: Zones }] : []),
            { path: Path.REPORTS, exact: true, component: Reports },
            ...(isShowUsersSection() ? [{ path: Path.USERS, exact: true, component: Users }] : []),
            ...(showTaxiCompanyRides() ? [{ path: Path.RIDES, exact: true, component: Rides }] : []),
            ...(isUserProfilePageFlag()
                ? [
                      { path: Path.USER_PROFILE, exact: true, component: UserProfile },
                      ...(isUserDocumentsPageFlag()
                          ? [
                                {
                                    path: Path.USER_DOCUMENTS,
                                    exact: true,
                                    component: UserDocuments,
                                },
                            ]
                          : []),
                      ...(isShowUsersRides() ? [{ path: Path.USER_RIDES, exact: true, component: UserRides }] : []),
                  ]
                : []),
            { path: Path.SETTINGS, exact: true, component: Settings },
            { path: Path.SETTINGS_GENERAL, exact: true, component: Settings },
            ...(isSettingsPricingFlag()
                ? [{ path: Path.SETTINGS_PRICING, exact: true, component: SettingsPricing }]
                : []),
            ...(isShowNotificationPageFlag()
                ? [{ path: Path.SETTINGS_NOTIFICATIONS, exact: true, component: SettingsNotifications }]
                : []),
            { path: Path.ERROR_404, exact: true, component: ErrorNotFoundView },
            { path: Path.ERROR_403, exact: true, component: ErrorPermissionsDenyView },
            {
                path: Path.HOME,
                exact: true,
                component: Portfolio,
                ...(!getDashboardComponent() && { redirect: Path.CARS }),
            },
        ];

        setRoutes(routes);

        if (!isVlootkit()) {
            let fav: any = document.querySelector("link[rel='shortcut icon']");
            let favhost = 'https://favicon.yandex.net/favicon/v2/';
            if (isBaseLang()) {
                fav.href = `${favhost}https://yandex.lt?size=32`;
            } else {
                fav.href = `${favhost}https://yandex.ru?size=32`;
            }
        }
    }, [flags]);

    return (
        <div className={style.content}>
            <CacheRequest.Provider value={new CacheRequestContext()}>
                <Router>
                    <NotificationCenterContext.Provider value={{ notificationsStore, notificationDispatch }}>
                        <FlagsContext.Provider value={flags}>
                            <CurrentUserContext.Provider value={currentUser}>
                                <GlobalTooltipContext.Provider
                                    value={{
                                        show,
                                        elementPosition,
                                        elementSize,
                                        setShow,
                                        setElementPosition,
                                        setElementSize,
                                        tooltipOptions,
                                        setTooltipOptions,
                                    }}
                                >
                                    <RouterScrollTop>
                                        <NotificationCenter />
                                        <SideMenu
                                            isLoading={isLoading}
                                            userInfo={userInfo}
                                        />
                                        {isLoading ? (
                                            <PageLoading />
                                        ) : (
                                            <>
                                                <GlobalSidebar />
                                                <GlobalTooltip
                                                    show={show}
                                                    elementPosition={elementPosition}
                                                    elementSize={elementSize}
                                                    tooltipOptions={tooltipOptions}
                                                />
                                                <Suspense fallback={<div />}>
                                                    <Switch>
                                                        {routes.map((route, index) => {
                                                            if (route.redirect) {
                                                                return (
                                                                    <Redirect
                                                                        key={route.path}
                                                                        strict={route.strict ?? false}
                                                                        exact={route.exact ?? false}
                                                                        from={route.redirectFrom ?? ''}
                                                                        to={route.redirect}
                                                                    />
                                                                );
                                                            } else {
                                                                return (
                                                                    <Route
                                                                        key={index}
                                                                        strict={route.strict ?? false}
                                                                        exact={route.exact ?? false}
                                                                        path={route.path}
                                                                        component={route.component}
                                                                    />
                                                                );
                                                            }
                                                        })}
                                                        <Route path="*">
                                                            <Redirect to={Path.ERROR_404} />
                                                        </Route>
                                                    </Switch>
                                                </Suspense>
                                            </>
                                        )}
                                    </RouterScrollTop>
                                </GlobalTooltipContext.Provider>
                            </CurrentUserContext.Provider>
                        </FlagsContext.Provider>
                    </NotificationCenterContext.Provider>
                </Router>
            </CacheRequest.Provider>
        </div>
    );
};

export default Content;
