import * as React from 'react';
import { useMemo, useState } from 'react';
import { Link } from 'react-router-dom';

import { UseGroupSignalFormResource } from 'entities/Car/api/useCarsGroups/useCarsGroups';
import { CarGroupNamedFilterSchema } from 'entities/Car/types/CarGroupSchema';

import { Label } from 'components/ui/Label';
import Select, { IOptionInfo } from 'components/ui/Select';

import { i18n } from 'components/GlobalSidebar/ModalAddSignal/GroupSignalForm/GroupSignalFormList/index.i18n';

import style from 'components/GlobalSidebar/ModalAddSignal/GroupSignalForm/GroupSignalFormList/index.css';

export interface GroupSignalFormListProps {
    resource: UseGroupSignalFormResource;
    onChange: (values: { named_filters: { include_dynamic_group: string[] } }) => void;
}

const mapper = (filter: CarGroupNamedFilterSchema): IOptionInfo => {
    return {
        value: filter.id,
        text: filter.display_name,
    };
};

const filterZone = (list: string[], element: string): string[] => {
    return list.filter((value) => value !== element);
};

export const GroupSignalFormList: React.FC<GroupSignalFormListProps> = function GroupSignalFormList({
    onChange,
    resource,
}) {
    const carsGroups = resource.read();

    const [groupsSignalSettings, setGroupsSignalSettings] = useState<string[]>([]);
    const onSelectSettingsChange = (groupKey: string) => {
        const nextValue = groupsSignalSettings.includes(groupKey)
            ? filterZone(groupsSignalSettings, groupKey)
            : [...groupsSignalSettings, groupKey];

        setGroupsSignalSettings(nextValue);

        onChange({ named_filters: { include_dynamic_group: nextValue } });
    };

    const data = useMemo<IOptionInfo[]>(() => {
        return [...carsGroups.self_named_filters.map(mapper), ...carsGroups.company_named_filters.map(mapper)];
    }, []);

    return (
        <>
            <div className={style.form_input}>
                {data && data.length ? (
                    <Select
                        onSelect={onSelectSettingsChange}
                        options={data}
                        placeholder={i18n('Choose groups')}
                        withSearch
                        isClosable
                    />
                ) : (
                    <>
                        <Select
                            onSelect={onSelectSettingsChange}
                            options={[]}
                            placeholder={i18n('No groups available')}
                            withSearch
                        />

                        <div className={style.info_message}>
                            {i18n('Groups have not been created yet, they can be configured in the section')}
                            <Link to="/cars/groups">{i18n('Groups')}</Link>
                        </div>
                    </>
                )}
            </div>
            <div className={style.form_label}>
                {groupsSignalSettings.map((value, index) => {
                    const option = data.find((option) => option.value === value);

                    if (!option || !option.text) {
                        return null;
                    }

                    return (
                        <Label
                            title={option.text}
                            background
                            bgClassName={style.form_label}
                            textClassName={style.form_label}
                            key={index}
                        />
                    );
                })}
            </div>
        </>
    );
};
