import React, { MutableRefObject, useEffect, useRef, useState } from 'react';
import { Link } from 'react-router-dom';

import { Spin } from 'shared/ui/Spin/Spin';

import NotificationCenterContext, { addNotification } from 'components/NotificationCenter/store';
import { NotificationIconType } from 'components/NotificationCenter/types';
import { Label } from 'components/ui/Label';
import Select from 'components/ui/Select';
import { REQUESTS, ZONES_REQUESTS } from 'components/Zones/request';

import { RequestHelper } from '../../../../../request-helper/src';

import { i18n } from 'components/GlobalSidebar/ModalAddSignal/ZoneSignalForm/index.i18n';

import style from 'components/GlobalSidebar/ModalAddSignal/ZoneSignalForm/index.css';

interface ZoneSignalProps {
    zoneFilter: string;
    onChange: (values: { zone: Record<string, string[]> }) => void;
}

const ZoneSignalForm = ({ zoneFilter, onChange }: ZoneSignalProps) => {
    const [zonesSignalSettings, setZonesSignalSettings] = useState<string[]>([]);
    const [zones, setZones] = useState<string[] | null>(null);
    const [isLoading, setLoading] = useState<boolean>(true);
    const [error, setError] = useState<Error | null>(null);

    const request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: ZONES_REQUESTS }));
    let { notificationDispatch } = React.useContext(NotificationCenterContext) || {};

    useEffect(() => {
        getZones();

        return () => {
            request.current.abort();
        };
    }, []);

    const getZones = () => {
        setLoading(true);
        setError(null);
        request.current
            .exec(REQUESTS.GET_ZONES)
            .finally(() => {
                setLoading(false);
            })
            .then((response) => {
                let zones = Object.keys(response?.objects?.company_zones) ?? [];
                setZones(zones);
            })
            .catch((error) => {
                setError(error);
                let notificationOptions = {
                    title: i18n('Failed to load geofences'),
                    iconType: NotificationIconType.ERROR,
                };

                notificationDispatch(addNotification(notificationOptions));
            });
    };

    function filterZone(list: string[], element: string) {
        return list.filter((value) => value !== element);
    }

    const onSelectSettingsChange = (zoneKey: string) => {
        const nextValue = zonesSignalSettings.includes(zoneKey)
            ? filterZone(zonesSignalSettings, zoneKey)
            : [...zonesSignalSettings, zoneKey];

        setZonesSignalSettings(nextValue);

        onChange({ zone: { [zoneFilter]: nextValue } });
    };

    return (
        <>
            <div className={style.form_input}>
                <div className={style.info}>
                    <span className={style.title}>{i18n('Geofences')}</span>
                    <span className={style.description}>{i18n('Choose geofence for vehicle leaving control')}</span>
                </div>
            </div>
            <div className={style.form_input}>
                {isLoading ? (
                    <Spin />
                ) : error ? (
                    <Select
                        onSelect={onSelectSettingsChange}
                        options={[]}
                        placeholder={i18n('No geofences available')}
                        withSearch
                    />
                ) : zones && zones.length ? (
                    <Select
                        onSelect={onSelectSettingsChange}
                        options={zones.map((value) => {
                            return { text: value, value: value };
                        })}
                        placeholder={i18n('Choose geofence')}
                        withSearch
                        isClosable
                    />
                ) : (
                    <>
                        <Select
                            onSelect={onSelectSettingsChange}
                            options={[]}
                            placeholder={i18n('No geofences available')}
                            withSearch
                        />

                        <div className={style.info_message}>
                            {i18n('Geofences have not been created yet, they can be configured in the section')}
                            <Link to="/signals/zones">{i18n('Geofences')}</Link>
                        </div>
                    </>
                )}
            </div>
            <div className={style.form_label}>
                {zonesSignalSettings.map((value, index) => {
                    return (
                        <Label
                            title={value}
                            background
                            bgClassName={style.form_label}
                            textClassName={style.form_label}
                            key={index}
                        />
                    );
                })}
            </div>
        </>
    );
};

export default ZoneSignalForm;
