import React, { MutableRefObject, ReactComponentElement, useContext, useRef, useState } from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import { MODAL_OBJECT_ID_CGI, MODAL_OBJECT_TYPE_CGI } from 'constants/constants';

import { isCarsGroupsAccess } from 'utils/isCarsGroupsAccess';

import { getSignalTypeDesc } from 'entities/Signal/helpers/getSignalTypeDesc/getSignalTypeDesc';
import { getSignalTypeTitle } from 'entities/Signal/helpers/getSignalTypeTitle/getSignalTypeTitle';
import { SignalType } from 'entities/Signal/types/SignalType';

import { Path } from 'shared/consts/Path';

import { GlobalSidebarHeader } from 'components/GlobalSidebar/GlobalSidebarHeader';
import CarBatterySignalForm from 'components/GlobalSidebar/ModalAddSignal/CarBatterySignalForm';
import { ChangeVinSignalForm } from 'components/GlobalSidebar/ModalAddSignal/ChangeVinSignalForm/ChangeVinSignalForm';
import {
    CUSTOMIZATION_STEP_NUMBER,
    SIGNALS,
    TYPE_SELECTION_STEP_NUMBER,
} from 'components/GlobalSidebar/ModalAddSignal/constants';
import DiangosticSignalForm from 'components/GlobalSidebar/ModalAddSignal/DiagnosticSignalForm';
import { EngineOverheatingSignalForm } from 'components/GlobalSidebar/ModalAddSignal/EngineOverheatingSignalForm/EngineOverheatingSignalForm';
import GroupSignalForm from 'components/GlobalSidebar/ModalAddSignal/GroupSignalForm';
import OvermileageSignalForm from 'components/GlobalSidebar/ModalAddSignal/OvermileageSignalForm';
import { MANAGE_SIGNALS_REQUESTS, REQUESTS } from 'components/GlobalSidebar/ModalAddSignal/request';
import SignalCard from 'components/GlobalSidebar/ModalAddSignal/SignalCard';
import SignalFormWithoutConfig from 'components/GlobalSidebar/ModalAddSignal/SignalFormWithoutConfig';
import SpeedingSignalForm from 'components/GlobalSidebar/ModalAddSignal/SpeedingSignalForm';
import TelematicsLagSignalForm from 'components/GlobalSidebar/ModalAddSignal/TelematicsLagSignalForm';
import ZoneSignalForm from 'components/GlobalSidebar/ModalAddSignal/ZoneSignalForm';
import { FormStep } from 'components/GlobalSidebar/ModalUpsertCarView/FormStep';
import NotificationCenterContext, { addNotification } from 'components/NotificationCenter/store';
import ControlButton from 'components/ui/Buttons/ControlButton';
import ErrorLabel from 'components/ui/ErrorLabel';
import { Input } from 'components/ui/Input';
import { RadioButton } from 'components/ui/RadioButton';
import Header2 from 'components/ui/Text/Header2';

import { RequestHelper } from '../../../../request-helper/src';

import { i18n } from 'components/GlobalSidebar/ModalAddSignal/index.i18n';

import style from 'components/GlobalSidebar/ModalAddSignal/index.css';

const ModalAddSignal = () => {
    const [activeStep, setActiveStep] = useState<number>(TYPE_SELECTION_STEP_NUMBER);
    const [selectedSignal, setSelectedSignal] = useState<SignalType | null>(null);
    const [signalPriority, setSignalPriority] = useState<string>('normal');
    const [signalName, setSignalName] = useState<string>('');
    const [isSignalCreating, setIsSignalCreating] = useState<boolean>(false);
    const [creatingError, setCreatingError] = useState<Error | null>(null);
    const hasCarsGroupsAccess = isCarsGroupsAccess();

    const [signalFormValues, setSignalFormValues] = useState<any>({});
    const [signalGroupsFormValues, setGroupsSignalFormValues] = useState<any>({});

    const request: MutableRefObject<RequestHelper> = useRef(
        new RequestHelper({ requestConfigs: MANAGE_SIGNALS_REQUESTS }),
    );

    let { notificationDispatch } = useContext(NotificationCenterContext) || {};

    let location = useLocation();
    let history = useHistory();

    const onCardClick = (signalId: SignalType) => {
        setSelectedSignal(signalId);
        setActiveStep(CUSTOMIZATION_STEP_NUMBER);
    };

    const onBackClick = () => {
        if (activeStep === CUSTOMIZATION_STEP_NUMBER) {
            setActiveStep(TYPE_SELECTION_STEP_NUMBER);
            setSignalName('');
            setSignalPriority('normal');
        }
    };

    const onSignalPriorityChange = (signalPriority: string) => {
        setSignalPriority(signalPriority);
    };

    const onCreateSignalClick = () => {
        setIsSignalCreating(true);
        setCreatingError(null);
        let mainInfo = {
            type: selectedSignal,
            display_name: signalName,
            signal_priority: signalPriority,
        };

        let data = Object.assign({}, mainInfo, signalFormValues, signalGroupsFormValues);
        request.current
            .exec(REQUESTS.ADD_SIGNAL, { body: { signal_configuration: data } })
            .finally(() => {
                setIsSignalCreating(false);
            })
            .then(() => {
                let searchParams = new URLSearchParams(location.search);
                searchParams.delete(MODAL_OBJECT_TYPE_CGI);
                searchParams.delete(MODAL_OBJECT_ID_CGI);
                history.push(`${location.pathname}?${searchParams}`);

                notificationDispatch(
                    addNotification({
                        title: i18n('Signal created'),
                        link: {
                            label: i18n('Go to the signal settings section'),
                            to: Path.SIGNALS_MANAGE,
                        },
                    }),
                );
            })
            .catch(setCreatingError);
    };

    let content: ReactComponentElement<any> | null = null;
    switch (activeStep) {
        case TYPE_SELECTION_STEP_NUMBER:
            content = (
                <div className={style.signals_lib}>
                    {SIGNALS.map((id) => {
                        const title = getSignalTypeTitle(id) || '???';
                        const description = getSignalTypeDesc(id);

                        return (
                            <SignalCard
                                key={id}
                                title={title}
                                description={description}
                                onClick={onCardClick.bind(null, id)}
                            />
                        );
                    })}
                </div>
            );

            break;
        case CUSTOMIZATION_STEP_NUMBER:
            let form: ReactComponentElement<any> | null = null;
            switch (selectedSignal) {
                case 'speeding_signal':
                    form = <SpeedingSignalForm onChange={setSignalFormValues} />;
                    break;
                case 'telematics_lag_signal':
                    form = <TelematicsLagSignalForm onChange={setSignalFormValues} />;
                    break;
                case 'car_battery_died_signal':
                    form = <CarBatterySignalForm onChange={setSignalFormValues} />;
                    break;
                case 'car_out_zone_signal':
                    form = (
                        <ZoneSignalForm
                            zoneFilter="excluded_group_ids"
                            onChange={setSignalFormValues}
                        />
                    );

                    break;
                case 'car_in_zone_signal':
                    form = (
                        <ZoneSignalForm
                            zoneFilter="included_group_ids"
                            onChange={setSignalFormValues}
                        />
                    );

                    break;
                case 'car_over_mileage_signal':
                    form = <OvermileageSignalForm onChange={setSignalFormValues} />;
                    break;
                case 'dashboard_check_engine_signal':
                case 'dashboard_check_airbag_signal':
                case 'dashboard_tyre_pressure_signal':
                case 'dashboard_washer_liquid_signal':
                case 'dashboard_check_oil_signal':
                case 'dashboard_check_brake_signal':
                case 'car_changed_mileage_signal':
                    form = <SignalFormWithoutConfig />;
                    break;
                case 'warning_dtc_error_signal':
                    form = <DiangosticSignalForm onChange={setSignalFormValues} />;
                    break;
                case 'engine_overheating_signal':
                    form = <EngineOverheatingSignalForm onChange={setSignalFormValues} />;
                    break;
                case 'change_vin_signal':
                    form = <ChangeVinSignalForm onChange={setSignalFormValues} />;
                    break;
            }

            content = (
                <div className={style.signal_main_info_form}>
                    <Input
                        value={signalName}
                        label={i18n('Signal name')}
                        onChange={setSignalName}
                    />

                    <div className={style.form_separator} />
                    <div className={style.critical_input}>
                        <span className={style.title}>{i18n('Signal priority')}</span>
                        <RadioButton
                            secondary
                            fullWidth
                            selectedValue={signalPriority}
                            values={[
                                {
                                    value: 'normal',
                                    placeholder: i18n('Normal'),
                                },

                                {
                                    value: 'critical',
                                    placeholder: i18n('Critical'),
                                },
                            ]}
                            onChange={onSignalPriorityChange}
                        />

                        <span className={style.description}>
                            {i18n(
                                'The signal priority affects the style of signal display in the signal center, as well as the signal sorting parameters.',
                            )}
                        </span>
                    </div>
                    <div className={style.form_separator} />
                    {hasCarsGroupsAccess && (
                        <>
                            <GroupSignalForm onChange={setGroupsSignalFormValues} />
                            <div className={style.form_separator} />
                        </>
                    )}

                    {form}
                    <div className={style.add_signal_button_container}>
                        {creatingError ? (
                            <div className={style.error_container}>
                                <ErrorLabel simple />
                            </div>
                        ) : null}
                        <ControlButton
                            fullWidth
                            title={i18n('Create signal')}
                            onClick={onCreateSignalClick}
                            disabled={!signalName || isSignalCreating}
                        />
                    </div>
                </div>
            );

            break;
    }

    return (
        <>
            <GlobalSidebarHeader backAction={onBackClick}>
                <div className={style.title}>
                    <Header2>{i18n('Creating a new signal')}</Header2>
                    {selectedSignal && activeStep === CUSTOMIZATION_STEP_NUMBER ? (
                        <span className={style.selected_signal}>{getSignalTypeTitle(selectedSignal)}</span>
                    ) : null}
                </div>
            </GlobalSidebarHeader>
            <div className={style.content_container}>
                <div className={style.steps_container}>
                    <div className={style.steps}>
                        <FormStep
                            step={TYPE_SELECTION_STEP_NUMBER}
                            title={i18n('Type selection')}
                            disabled={activeStep !== TYPE_SELECTION_STEP_NUMBER}
                        />

                        <div className={style.separator} />
                        <FormStep
                            step={CUSTOMIZATION_STEP_NUMBER}
                            title={i18n('Settings')}
                            disabled={activeStep !== CUSTOMIZATION_STEP_NUMBER}
                        />
                    </div>
                    {activeStep == TYPE_SELECTION_STEP_NUMBER ? (
                        <div className={style.advice}>
                            {i18n('The selected signal type determines the settings that can be applied to it.')}
                        </div>
                    ) : null}
                </div>
                <div className={style.content}>{content}</div>
            </div>
        </>
    );
};

export default ModalAddSignal;
