import * as React from 'react';
import { MutableRefObject, useRef, useState } from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import { MODAL_OBJECT_ID_CGI, MODAL_OBJECT_TYPE_CGI } from 'constants/constants';

import { Path } from 'shared/consts/Path';

import { GlobalSidebarHeader } from 'components/GlobalSidebar/GlobalSidebarHeader';
import UserForm, { PHONE_START_SYMBOL } from 'components/GlobalSidebar/ModalUserView/UserForm';
import formatPhoneValue from 'components/GlobalSidebar/ModalUserView/UserForm/formatPhoneValue';
import NotificationCenterContext, { addNotification } from 'components/NotificationCenter/store';
import { NotificationIconType } from 'components/NotificationCenter/types';
import ControlButton from 'components/ui/Buttons/ControlButton';
import Header2 from 'components/ui/Text/Header2';
import { REQUESTS, USERS_REQUESTS } from 'components/Users/request';

import { RequestHelper } from '../../../../request-helper/src';

import { i18n } from 'components/GlobalSidebar/ModalAddUserView/index.i18n';

import style from 'components/GlobalSidebar/ModalAddUserView/index.css';

const ModalAddUserView = () => {
    const [userData, setUserData] = useState<{
        lastName: string;
        firstName: string;
        pName: string;
        phone: string;
        email: string;
    }>({
        lastName: '',
        firstName: '',
        pName: '',
        phone: '',
        email: '',
    });

    const [isCreating, setIsCreating] = useState<boolean>(false);
    const [isPhoneNumberExist, setIsPhoneNumberExist] = useState<boolean>(false);
    let { notificationDispatch } = React.useContext(NotificationCenterContext) || {};
    const request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: USERS_REQUESTS }));

    const location = useLocation();
    const history = useHistory();

    const onCreateClick = () => {
        setIsCreating(true);
        setIsPhoneNumberExist(false);

        let data = {
            last_name: userData.lastName,
            first_name: userData.firstName,
            pn: userData.pName,
            phone_number: userData.phone,
            email: userData.email,
        };

        let searchPhoneValue = formatPhoneValue(data.phone_number);

        request.current
            .exec(REQUESTS.SEARCH, { queryParams: { has_all_of: searchPhoneValue } })
            .then((response) => {
                let isUserExist = response?.objects?.users.filter((user) => user.status !== 'deleted')?.length;

                if (isUserExist) {
                    setIsPhoneNumberExist(true);
                    setIsCreating(false);
                } else {
                    request.current
                        .exec(REQUESTS.ADD_USER, { body: data })
                        .then(() => {
                            setIsCreating(false);

                            let searchParams = new URLSearchParams(location.search);
                            searchParams.delete(MODAL_OBJECT_TYPE_CGI);
                            searchParams.delete(MODAL_OBJECT_ID_CGI);
                            history.push(`${location.pathname}?${searchParams}`);

                            let notificationOptions = {
                                title: i18n('User is created'),
                                link: {
                                    label: i18n('Go to Users section'),
                                    to: Path.USERS,
                                },
                            };

                            notificationDispatch(addNotification(notificationOptions));
                        })
                        .catch(() => {
                            setIsCreating(false);

                            let notificationOptions = {
                                title: i18n('Failed to create user'),
                                iconType: NotificationIconType.ERROR,
                            };

                            notificationDispatch(addNotification(notificationOptions));
                        });
                }
            })
            .catch(() => {
                setIsCreating(false);

                let notificationOptions = {
                    title: i18n('Failed to create user'),
                    iconType: NotificationIconType.ERROR,
                };

                notificationDispatch(addNotification(notificationOptions));
            });
    };

    const onUserFormDataChange = (userFormData) => {
        if (userFormData.phone !== userData.phone) {
            setIsPhoneNumberExist(false);
        }
        setUserData(userFormData);
    };

    let isRequiredFieldsEntered =
        userData.lastName && userData.firstName && userData.phone.replace(PHONE_START_SYMBOL, '');

    return (
        <div>
            <GlobalSidebarHeader>
                <Header2>{i18n('Add driver')}</Header2>
            </GlobalSidebarHeader>
            <div className={style.content_container}>
                <UserForm
                    onChange={onUserFormDataChange}
                    errors={{ phone: isPhoneNumberExist ? i18n('User with this phone number already exists') : '' }}
                />

                <div className={style.separator} />
                <div className={style.roles}>
                    <div className={style.title}>
                        <span className={style.main}>{i18n('User roles')}</span>
                        <span className={style.description}>
                            {i18n('Responsible for access to the admin panel and application')}
                        </span>
                    </div>
                    <div className={style.roles_list}>
                        <div className={style.role_label}>{i18n('Driver')}</div>
                    </div>
                </div>
                <div className={style.create_button_container}>
                    <ControlButton
                        fullWidth
                        disabled={!isRequiredFieldsEntered}
                        isLoading={isCreating}
                        title={i18n('Add driver')}
                        onClick={onCreateClick}
                    />
                </div>
            </div>
        </div>
    );
};

export default ModalAddUserView;
