import React, { MutableRefObject, useRef, useState } from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import { EMPTY_DATA, MODAL_OBJECT_ID_CGI, MODAL_OBJECT_TYPE_CGI } from 'constants/constants';

import { CARS_GROUPS_REQUESTS, REQUESTS } from 'components/Cars/CarsGroups/request';
import { IGroup } from 'components/Cars/CarsGroups/types';
import Confirm from 'components/ui/Confirm';

import { RequestHelper } from '../../../../../request-helper/src';

import { i18n } from 'components/GlobalSidebar/ModalGroupView/DeleteGroupConfirm/index.i18n';

import style from 'components/GlobalSidebar/ModalGroupView/DeleteGroupConfirm/index.css';

interface IDeleteGroupConfirmProps {
    onClose: () => void;
    group: IGroup | null;
    cars: any[];
}

const DeleteGroupConfirm = ({ group, onClose, cars }: IDeleteGroupConfirmProps) => {
    const request: MutableRefObject<RequestHelper> = useRef(
        new RequestHelper({ requestConfigs: CARS_GROUPS_REQUESTS }),
    );

    const [isLoading, setIsLoading] = useState<boolean>(false);
    const [error, setError] = useState<Error | null>(null);

    let location = useLocation();
    let history = useHistory();

    const onDeleteClick = () => {
        setIsLoading(true);
        setError(null);
        request.current
            .exec(REQUESTS.REMOVE_GROUP, { body: { ids: [group?.id] } })
            .finally(() => {
                setIsLoading(false);
            })
            .then(() => {
                onClose();

                let searchParams = new URLSearchParams(location.search);
                searchParams.delete(MODAL_OBJECT_TYPE_CGI);
                searchParams.delete(MODAL_OBJECT_ID_CGI);
                history.push(`${location.pathname}?${searchParams}`);
            })
            .catch(setError);
    };

    return (
        <Confirm
            title={i18n('Are you sure you want to delete the group?')}
            description={
                <div className={style.info}>
                    <div className={style.group_info}>
                        <span>{group?.display_name || EMPTY_DATA}</span>
                        <span className={style.count}>{`${i18n('Cars:')} ${cars.length}`}</span>
                    </div>
                    <span>{i18n('All cars will lose their connection to this group')}</span>
                    <div className={style.separator} />
                </div>
            }
            confirmButtonTitle={i18n('Yes, delete')}
            closeButtonTitle={i18n('Cancel')}
            onConfirm={onDeleteClick}
            onClose={onClose}
            isLoading={isLoading}
            error={error}
        />
    );
};

export default DeleteGroupConfirm;
