import React, { MutableRefObject, useEffect, useRef, useState } from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import { EMPTY_DATA, MODAL_OBJECT_ID_CGI, MODAL_OBJECT_TYPE_CGI } from 'constants/constants';

import { showOrganizationGroupInfo } from 'utils/showOrganizationGroupInfo';

import { IGroupModalErrorField, IGroupModalFormField } from 'components/Cars/CarsGroups/GroupModal';
import { CARS_GROUPS_REQUESTS, REQUESTS } from 'components/Cars/CarsGroups/request';
import { IGroup } from 'components/Cars/CarsGroups/types';
import { CARS_REQUESTS, REQUESTS as CAR_REQUESTS } from 'components/Cars/request';
import { GlobalSidebarHeader } from 'components/GlobalSidebar/GlobalSidebarHeader';
import DeleteGroupConfirm from 'components/GlobalSidebar/ModalGroupView/DeleteGroupConfirm';
import SearchResults from 'components/GlobalSidebar/ModalSearchView/SearchResults';
import CommandButton from 'components/ui/Buttons/CommandButton';
import ControlButton from 'components/ui/Buttons/ControlButton';
import ErrorLabel from 'components/ui/ErrorLabel';
import ErrorReloadLabel from 'components/ui/ErrorLabel/ErrorReloadLabel';
import { Input } from 'components/ui/Input';
import Tabs from 'components/ui/Tabs';
import Header2 from 'components/ui/Text/Header2';

import { RequestHelper } from '../../../../request-helper/src';

import { i18n } from 'components/GlobalSidebar/ModalGroupView/index.i18n';

import coreStyle from 'components/Content/index.css';
import style from 'components/GlobalSidebar/ModalGroupView/index.css';

const ModalGroupView = () => {
    const [group, setGroup] = useState<IGroup | null>(null);
    const [isLoading, setIsLoading] = useState<boolean>(false);
    const [error, setError] = useState<Error | null>(null);

    const [isCarsSearching, setIsCarsSearching] = useState<boolean>(false);
    const [carsSearchingError, setCarsSearchingError] = useState<Error | null>(null);
    const [cars, setCars] = useState<any[]>([]);

    const [isUpserting, setIsUpserting] = useState<boolean>(false);
    const [upsertingError, setUpsertingError] = useState<Error | null>(null);

    const [currentTab, setCurrentTab] = useState<string>('cars');
    const [isDeleteConfirmOpen, setIsDeleteConfirmOpen] = useState<boolean>(false);
    const [formValue, setFormValue] = useState<IGroupModalFormField>({
        display_name: '',
        description: '',
    });

    const [formErrors, setFormErrors] = useState<IGroupModalErrorField>({
        display_name: '',
        description: '',
    });

    const request: MutableRefObject<RequestHelper> = useRef(
        new RequestHelper({ requestConfigs: CARS_GROUPS_REQUESTS }),
    );

    const requestCars: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: CARS_REQUESTS }));

    let location = useLocation();
    let history = useHistory();

    let searchParams = new URLSearchParams(location.search);
    let groupId = searchParams.get(MODAL_OBJECT_ID_CGI);

    const TABS = [
        {
            link: 'cars',
            name: i18n('Cars'),
        },

        {
            link: 'settings',
            name: i18n('Settings'),
        },
    ];

    useEffect(() => {
        getGroups(groupId).then((groups) => {
            let selectedGroup = groups[0];
            setGroup(selectedGroup);
            setFormValue({
                display_name: selectedGroup?.display_name ?? '',
                description: selectedGroup?.description ?? '',
            });
        });
    }, []);

    useEffect(() => {
        if (group) {
            getGroupCars();
        }
    }, [group]);

    const getGroups = (id?: string | null) => {
        if (id) {
            setIsLoading(true); //show shimmers only for loading one current group
        }
        setError(null);

        return request.current
            .exec(REQUESTS.GET_GROUPS, { queryParams: { id: id ?? null } })
            .finally(() => {
                setIsLoading(false);
            })
            .then((response: { objects: { company_named_filters: IGroup[]; self_named_filters: IGroup[] } }) => {
                return (
                    [
                        ...(response.objects.company_named_filters ?? []).map((group) => {
                            group.visibility = 'company';

                            return group;
                        }),
                        ...(response.objects.self_named_filters ?? []).map((group) => {
                            group.visibility = 'user';

                            return group;
                        }),
                    ] ?? []
                );
            })
            .catch(setError);
    };

    const getGroupCars = () => {
        setIsCarsSearching(true);
        setCarsSearchingError(null);

        return requestCars.current
            .exec(CAR_REQUESTS.GET_CARS, {
                queryParams: {
                    tags_filter: group?.filter?.conditions ?? '',
                    traits: 'ReportVIN',
                },
            })
            .finally(() => {
                setIsCarsSearching(false);
            })
            .then((response) => {
                let cars = response?.cars ?? [];
                setCars(cars);
            })
            .catch(setCarsSearchingError);
    };

    const onTabChange = (currentTab: string) => {
        setCurrentTab(currentTab);
    };

    const onFormChange = (type: string, value: string) => {
        let newFormValue = Object.assign({}, formValue);
        newFormValue[type] = value;

        setFormValue(newFormValue);

        let newFormErrors = Object.assign({}, formErrors);
        newFormErrors[type] = null;
        setFormErrors(newFormErrors);
    };

    const onUpsertClick = () => {
        setIsUpserting(true);
        setUpsertingError(null);
        setFormErrors({
            display_name: '',
            description: '',
        });

        let data: { objects: IGroup[] } = {
            objects: [
                {
                    id: group?.id,
                    display_name: formValue.display_name,
                    description: formValue.description,
                    filter: group?.filter ?? {
                        conditions: '',
                    },

                    priority: group?.priority,
                    revision: group?.revision,
                    type: group?.type,
                },
            ],
        };

        getGroups()
            .then((groups: IGroup[]) => {
                let isDisplayNameExist = groups.some(
                    (groupItem) =>
                        groupItem.id !== groupId &&
                        groupItem.display_name === formValue.display_name &&
                        groupItem.visibility === group?.visibility,
                );

                if (isDisplayNameExist) {
                    let newFormErrors = Object.assign({}, formErrors);
                    newFormErrors.display_name = i18n('A group with the same name already exists');

                    setFormErrors(newFormErrors);
                    setIsUpserting(false);
                } else {
                    request.current
                        .exec(REQUESTS.UPSERT_GROUP, { body: data })
                        .finally(() => {
                            setIsUpserting(false);
                        })
                        .then(() => {
                            let searchParams = new URLSearchParams(location.search);
                            searchParams.delete(MODAL_OBJECT_TYPE_CGI);
                            searchParams.delete(MODAL_OBJECT_ID_CGI);
                            history.push(`${location.pathname}?${searchParams}`);
                        })
                        .catch(setUpsertingError);
                }
            })
            .catch((error) => {
                setIsLoading(false);
                setError(error);
            });
    };

    const onDeleteClick = () => {
        setIsDeleteConfirmOpen(true);
    };

    const onDeleteConfirmClose = () => {
        setIsDeleteConfirmOpen(false);
    };

    const isFormChanged =
        group?.display_name !== formValue.display_name || group?.description !== formValue.description;

    let { type } = group ?? {};
    let typeDisplayName: string = '';
    if (type === 'car') {
        typeDisplayName = i18n('Cars');
    }

    let isUserHaveAccessToEdit = group?.visibility === 'company' ? showOrganizationGroupInfo() : true;

    return (
        <>
            <GlobalSidebarHeader>
                <div className={style.header}>
                    {isLoading ? (
                        <>
                            <div className={`${coreStyle.shimmer} ${style.title_shimmer}`} />
                            <div className={`${coreStyle.shimmer} ${style.type_shimmer}`} />
                        </>
                    ) : (
                        <>
                            <Header2>{group?.display_name || EMPTY_DATA}</Header2>
                            <span className={style.types}>{typeDisplayName}</span>
                        </>
                    )}
                </div>
            </GlobalSidebarHeader>
            {error ? (
                <ErrorReloadLabel reloadFunction={getGroups.bind(null, groupId)} />
            ) : (
                <div className={style.content}>
                    <div className={style.description}>
                        <div className={style.title}>{i18n('Description')}</div>
                        {isLoading ? (
                            <div className={`${coreStyle.shimmer} ${style.description_shimmer}`} />
                        ) : (
                            <div className={style.text}>{group?.description || EMPTY_DATA}</div>
                        )}
                    </div>
                    <Tabs
                        tabs={TABS}
                        onChange={onTabChange}
                        activeTab={currentTab}
                    />

                    {currentTab === 'cars' ? (
                        <div className={style.cars}>
                            {carsSearchingError ? (
                                <div className={style.empty_result}>
                                    <ErrorReloadLabel reloadFunction={getGroupCars} />
                                </div>
                            ) : isCarsSearching || cars.length ? (
                                <SearchResults
                                    withoutTypeLabels
                                    searchResults={cars}
                                    isSearching={isCarsSearching}
                                />
                            ) : (
                                <div className={style.empty_result}>{i18n('No cars found')}</div>
                            )}
                        </div>
                    ) : (
                        <div className={style.inputs}>
                            <div className={style.input}>
                                <Input
                                    value={formValue.display_name}
                                    label={i18n('Group name')}
                                    disabled={!isUserHaveAccessToEdit || isLoading || isUpserting}
                                    error={formErrors.display_name}
                                    onChange={onFormChange.bind(null, 'display_name')}
                                />
                            </div>
                            <div className={style.separator} />
                            <div className={style.input}>
                                <Input
                                    value={formValue.description}
                                    label={i18n('Description')}
                                    disabled={!isUserHaveAccessToEdit || isLoading || isUpserting}
                                    error={formErrors.description}
                                    onChange={onFormChange.bind(null, 'description')}
                                />
                            </div>
                            <div className={style.separator} />
                            <div className={style.control_label}>
                                <CommandButton
                                    critical
                                    title={i18n('Delete group')}
                                    onClick={onDeleteClick}
                                    disabled={!isUserHaveAccessToEdit || isLoading || isUpserting}
                                    description={i18n('All cars will lose their connection to this group')}
                                />
                            </div>
                            {isFormChanged ? (
                                <div className={style.save_button_container}>
                                    {upsertingError ? (
                                        <div className={style.error_label_container}>
                                            <ErrorLabel simple />
                                        </div>
                                    ) : null}
                                    <ControlButton
                                        title={i18n('Save changes')}
                                        onClick={onUpsertClick}
                                        disabled={isLoading || isUpserting}
                                        fullWidth
                                    />
                                </div>
                            ) : null}
                        </div>
                    )}
                </div>
            )}

            {isDeleteConfirmOpen ? (
                <DeleteGroupConfirm
                    group={group}
                    cars={cars}
                    onClose={onDeleteConfirmClose}
                />
            ) : null}
        </>
    );
};

export default ModalGroupView;
