import React, { MutableRefObject, useEffect, useRef, useState } from 'react';

import { i18n } from 'components/GlobalSidebar/ModalSearchView/SearchInput/index.i18n';

import ClearIcon from 'components/GlobalSidebar/ModalSearchView/SearchInput/clear_icon.component.svg';
import SearchIcon from 'components/GlobalSidebar/ModalSearchView/SearchInput/search_icon.component.svg';

import style from 'components/GlobalSidebar/ModalSearchView/SearchInput/index.css';

interface ISearchInputProps {
    value: string;
    onChange: (value: string) => void;
    searchIcon?: boolean;
    placeholder?: string;
    className?: string;
    clearIcon?: boolean;
    autoFocus?: boolean;
}

const SearchInput = ({
    value,
    onChange: onChangeProps,
    searchIcon,
    placeholder,
    className,
    clearIcon,
    autoFocus,
}: ISearchInputProps) => {
    const [focused, setFocused] = useState<boolean>(false);

    const inputRef: MutableRefObject<HTMLInputElement | null> = useRef(null);

    useEffect(() => {
        if (autoFocus) {
            inputRef?.current?.focus();
        }
    }, []);

    const onContainerFocus = () => {
        inputRef?.current?.focus();
        setFocused(true);
    };

    const onContainerBlur = () => {
        setFocused(false);
    };

    const onChange = (event: any) => {
        let value = event.target.value;
        onChangeProps(value);
    };

    return (
        <div
            className={`${style.input_container}` + ` ${focused ? style.focused : ''}` + ` ${className ?? ''}`}
            tabIndex={0}
            onFocus={onContainerFocus}
        >
            {searchIcon ? <SearchIcon /> : null}
            <input
                className={style.input}
                ref={inputRef}
                value={value}
                onBlur={onContainerBlur}
                placeholder={placeholder ?? i18n('Search by number, VIN or model')}
                onChange={onChange}
            />

            {clearIcon ? (
                <div
                    className={`${style.clear_icon} ${!value ? style.hidden : ''}`}
                    onClick={onChangeProps.bind(null, '')}
                >
                    <ClearIcon />
                </div>
            ) : null}
        </div>
    );
};

export default SearchInput;
